//+------------------------------------------------------------------+
//|                                                 ATLAS_Config.mqh |
//|                                    Copyright 2026, ATLAS FX Ltd. |
//|                                        https://www.atlasfxsignals.com   |
//+------------------------------------------------------------------+
#property copyright "Copyright 2026, ATLAS FX Ltd."
#property link      "https://www.atlasfxsignals.com"
#property strict

#ifndef ATLAS_CONFIG_MQH
#define ATLAS_CONFIG_MQH

//+------------------------------------------------------------------+
//| EA version                                                        |
//+------------------------------------------------------------------+
#define EA_VERSION        "1.0.0"
#define EA_NAME           "ATLAS EA"
#define EA_USER_AGENT     "ATLAS-EA/1.0 (MT4)"

//+------------------------------------------------------------------+
//| Input parameters — user-configurable in EA settings               |
//+------------------------------------------------------------------+
extern string   InpApiKey          = "";                              // API Key (ak_...)
extern string   InpServerUrl       = "https://api.atlasfxsignals.com";      // API Server URL

extern int      InpMinConfidence   = 60;                              // Minimum signal confidence (0-100)
extern double   InpMaxRiskPercent  = 1.0;                             // Risk per trade (% of balance)
extern double   InpMaxDailyDD     = 3.0;                              // Max daily drawdown (%)
extern int      InpMaxOpenTrades   = 3;                               // Max simultaneous positions
extern double   InpMaxSlippagePips = 1000.0;                           // Max entry price deviation (pips)

extern bool     InpAutoExecute     = false;                           // Auto-execute signals
extern int      InpMagicNumber     = 20260224;                        // EA magic number
extern int      InpPollInterval    = 10;                              // Polling interval (seconds)
extern string   InpSymbolSuffix    = "";                              // Broker symbol suffix (e.g., ".m")

extern bool     InpShowPanel       = true;                            // Show status panel on chart
extern int      InpPanelX          = 10;                              // Panel X position
extern int      InpPanelY          = 30;                              // Panel Y position
extern color    InpPanelBg         = clrMidnightBlue;                 // Panel background color
extern color    InpPanelText       = clrWhite;                        // Panel text color

//+------------------------------------------------------------------+
//| HTTP configuration constants                                      |
//+------------------------------------------------------------------+
#define HTTP_TIMEOUT_MS     10000
#define HTTP_MAX_RETRIES    1
#define HTTP_RETRY_DELAY_MS 1000

//+------------------------------------------------------------------+
//| Signal processing constants                                       |
//+------------------------------------------------------------------+
#define MAX_PROCESSED_SIGNALS 500
#define SIGNAL_ID_PREFIX      "sig_"

//+------------------------------------------------------------------+
//| Settings struct — populated from auth/validate response           |
//+------------------------------------------------------------------+
struct AtlasSettings
{
    int    min_confidence;
    double max_risk_pct;
    double max_daily_dd_pct;
    double max_weekly_dd_pct;
    int    max_open_trades;
    string allowed_symbols[];
    double max_slippage_pips;
    double max_entry_deviation_pct;
    bool   auto_execute;
};

void SettingsReset(AtlasSettings &s)
{
    s.min_confidence         = InpMinConfidence;
    s.max_risk_pct           = InpMaxRiskPercent;
    s.max_daily_dd_pct       = InpMaxDailyDD;
    s.max_weekly_dd_pct      = 5.0;
    s.max_open_trades        = InpMaxOpenTrades;
    ArrayResize(s.allowed_symbols, 1);
    s.allowed_symbols[0]     = "XAUUSD";
    s.max_slippage_pips      = InpMaxSlippagePips;
    s.max_entry_deviation_pct = 0.5;
    s.auto_execute           = InpAutoExecute;
}

//+------------------------------------------------------------------+
//| Signal data struct                                                |
//+------------------------------------------------------------------+
struct AtlasSignal
{
    string   id;
    string   symbol;
    string   direction;
    int      confidence;
    string   regime;
    double   entry_price;
    double   stop_price;
    double   target_price;
    double   risk_pips;
    double   reward_pips;
    double   rr_ratio;
    string   status;
    datetime expires_at;
    datetime created_at;
};

//+------------------------------------------------------------------+
//| HTTP response struct                                              |
//+------------------------------------------------------------------+
struct HttpResponse
{
    int    status_code;
    string body;
    string headers;
    bool   success;
    string error_message;
};

void HttpResponseReset(HttpResponse &r)
{
    r.status_code   = 0;
    r.body          = "";
    r.headers       = "";
    r.success       = false;
    r.error_message = "";
}

//+------------------------------------------------------------------+
//| Helper: Build API endpoint URL                                    |
//+------------------------------------------------------------------+
string BuildApiUrl(string endpoint)
{
    string base = InpServerUrl;
    if(StringGetCharacter(base, StringLen(base) - 1) == '/')
        base = StringSubstr(base, 0, StringLen(base) - 1);
    if(StringFind(base, "/v1") < 0)
        base = base + "/v1";
    return base + endpoint;
}

//+------------------------------------------------------------------+
//| Helper: Build broker symbol from signal symbol                    |
//+------------------------------------------------------------------+
string BuildBrokerSymbol(string signal_symbol)
{
    return signal_symbol + InpSymbolSuffix;
}

//+------------------------------------------------------------------+
//| Helper: Parse ISO 8601 datetime string to MQL4 datetime          |
//+------------------------------------------------------------------+
datetime ParseIso8601(string iso_str)
{
    if(StringLen(iso_str) < 19)
        return 0;

    string date_part = StringSubstr(iso_str, 0, 10);
    string time_part = StringSubstr(iso_str, 11, 8);

    StringReplace(date_part, "-", ".");
    string datetime_str = date_part + " " + time_part;
    return StrToTime(datetime_str);
}

//+------------------------------------------------------------------+
//| Helper: Format datetime to ISO 8601 string                       |
//+------------------------------------------------------------------+
string FormatIso8601(datetime dt)
{
    return StringFormat("%04d-%02d-%02dT%02d:%02d:%02dZ",
                        TimeYear(dt), TimeMonth(dt), TimeDay(dt),
                        TimeHour(dt), TimeMinute(dt), TimeSeconds(dt));
}

#endif // ATLAS_CONFIG_MQH
