//+------------------------------------------------------------------+
//|                                                    ATLAS_EA.mq4  |
//|                                    Copyright 2026, ATLAS FX Ltd. |
//|                                        https://www.atlasfxsignals.com   |
//+------------------------------------------------------------------+
#property copyright "Copyright 2026, ATLAS FX Ltd."
#property link      "https://www.atlasfxsignals.com"
#property version   "1.00"
#property description "ATLAS FX Expert Advisor — MT4 version"
#property strict

//+------------------------------------------------------------------+
//| Includes                                                          |
//+------------------------------------------------------------------+
#include "ATLAS_Config.mqh"
#include "ATLAS_Json.mqh"
#include "ATLAS_Http.mqh"
#include "ATLAS_Risk.mqh"
#include "ATLAS_Panel.mqh"

//+------------------------------------------------------------------+
//| Globals                                                           |
//+------------------------------------------------------------------+
AtlasHttp     g_http;
AtlasRisk     g_risk;
AtlasPanel    g_panel;
AtlasSettings g_settings;

string g_processed_ids[];
int    g_processed_count = 0;

bool     g_authenticated = false;
string   g_user_plan = "";
string   g_user_status = "";
datetime g_last_poll = 0;
int      g_signal_count = 0;
int      g_poll_errors = 0;

//+------------------------------------------------------------------+
//| Check if a signal ID has already been processed                   |
//+------------------------------------------------------------------+
bool IsSignalProcessed(string signal_id)
{
    for(int i = 0; i < g_processed_count; i++)
    {
        if(g_processed_ids[i] == signal_id)
            return true;
    }
    return false;
}

//+------------------------------------------------------------------+
//| Mark a signal ID as processed                                     |
//+------------------------------------------------------------------+
void MarkSignalProcessed(string signal_id)
{
    if(g_processed_count >= MAX_PROCESSED_SIGNALS)
    {
        int keep = MAX_PROCESSED_SIGNALS / 2;
        for(int i = 0; i < keep; i++)
            g_processed_ids[i] = g_processed_ids[g_processed_count - keep + i];
        g_processed_count = keep;
    }

    ArrayResize(g_processed_ids, g_processed_count + 1);
    g_processed_ids[g_processed_count] = signal_id;
    g_processed_count++;
}

//+------------------------------------------------------------------+
//| Execute a trade for a signal (MT4 OrderSend)                      |
//+------------------------------------------------------------------+
bool ExecuteSignal(AtlasSignal &signal)
{
    string broker_symbol = BuildBrokerSymbol(signal.symbol);

    // Check if symbol is valid
    double bid = MarketInfo(broker_symbol, MODE_BID);
    if(bid <= 0)
    {
        Print("[ATLAS EA] Symbol ", broker_symbol, " not available");
        g_http.SendFeedback(signal.id, "error", 0, 0, 0, 0, 0,
                            "Symbol not found: " + broker_symbol);
        return false;
    }

    double ask = MarketInfo(broker_symbol, MODE_ASK);
    int digits = (int)MarketInfo(broker_symbol, MODE_DIGITS);

    double current_price;
    int op_type;
    if(signal.direction == "buy")
    {
        current_price = ask;
        op_type = OP_BUY;
    }
    else
    {
        current_price = bid;
        op_type = OP_SELL;
    }

    // Check slippage
    if(!g_risk.IsSlippageAcceptable(broker_symbol, signal.entry_price,
                                     current_price, g_settings.max_slippage_pips))
    {
        Print("[ATLAS EA] Slippage too high for ", signal.id);
        g_http.SendFeedback(signal.id, "filtered", 0, 0, 0, 0, 0,
                            "", "slippage_exceeded");
        return false;
    }

    // Calculate lot size
    double lots = g_risk.CalculateLotSize(broker_symbol, current_price,
                                           signal.stop_price,
                                           g_settings.max_risk_pct);
    if(lots <= 0)
    {
        Print("[ATLAS EA] Cannot calculate lot size for ", signal.id);
        g_http.SendFeedback(signal.id, "error", 0, 0, 0, 0, 0,
                            "Position size too small");
        return false;
    }

    // Adjust SL/TP for actual entry price
    double adjusted_sl, adjusted_tp;
    g_risk.AdjustSlTp(signal.entry_price, current_price,
                       signal.stop_price, signal.target_price,
                       adjusted_sl, adjusted_tp);

    adjusted_sl = NormalizeDouble(adjusted_sl, digits);
    adjusted_tp = NormalizeDouble(adjusted_tp, digits);

    int slippage_points = (int)(g_settings.max_slippage_pips * 10);
    string comment = "ATLAS|" + signal.id;

    // MT4 OrderSend
    int ticket = OrderSend(broker_symbol, op_type, lots,
                           NormalizeDouble(current_price, digits),
                           slippage_points, adjusted_sl, adjusted_tp,
                           comment, InpMagicNumber, 0, clrNONE);

    if(ticket < 0)
    {
        int err = GetLastError();
        Print("[ATLAS EA] OrderSend failed for ", signal.id, ": error ", err);
        g_http.SendFeedback(signal.id, "error", 0, 0, 0, 0, 0,
                            "OrderSend error " + IntegerToString(err));
        return false;
    }

    // Get actual fill price
    double fill_price = current_price;
    if(OrderSelect(ticket, SELECT_BY_TICKET))
        fill_price = OrderOpenPrice();

    Print("[ATLAS EA] Executed ", signal.id, " ", signal.direction,
          " ", DoubleToString(lots, 2), " lots @ ", DoubleToString(fill_price, digits));

    g_http.SendFeedback(signal.id, "accepted", fill_price, lots);

    return true;
}

//+------------------------------------------------------------------+
//| Process signals from API response                                 |
//+------------------------------------------------------------------+
void ProcessSignals(AtlasSignal &signals[], int count)
{
    for(int i = 0; i < count; i++)
    {
        if(IsSignalProcessed(signals[i].id))
            continue;

        if(signals[i].status != "active")
        {
            MarkSignalProcessed(signals[i].id);
            continue;
        }

        if(g_risk.IsSignalExpired(signals[i].expires_at))
        {
            Print("[ATLAS EA] Signal ", signals[i].id, " expired");
            MarkSignalProcessed(signals[i].id);
            continue;
        }

        if(signals[i].confidence < g_settings.min_confidence)
        {
            g_http.SendFeedback(signals[i].id, "filtered", 0, 0, 0, 0, 0,
                                "", "low_confidence");
            MarkSignalProcessed(signals[i].id);
            continue;
        }

        if(!g_risk.IsSymbolAllowed(signals[i].symbol, g_settings.allowed_symbols))
        {
            g_http.SendFeedback(signals[i].id, "filtered", 0, 0, 0, 0, 0,
                                "", "symbol_not_allowed");
            MarkSignalProcessed(signals[i].id);
            continue;
        }

        int open_count = g_risk.CountOpenPositions(InpMagicNumber);
        if(open_count >= g_settings.max_open_trades)
        {
            g_http.SendFeedback(signals[i].id, "filtered", 0, 0, 0, 0, 0,
                                "", "max_trades_reached");
            MarkSignalProcessed(signals[i].id);
            continue;
        }

        if(g_risk.IsDailyDrawdownBreached(g_settings.max_daily_dd_pct))
        {
            g_http.SendFeedback(signals[i].id, "filtered", 0, 0, 0, 0, 0,
                                "", "daily_drawdown_limit");
            MarkSignalProcessed(signals[i].id);
            continue;
        }

        string broker_symbol = BuildBrokerSymbol(signals[i].symbol);
        if(!g_risk.IsMarketOpen(broker_symbol))
            continue;

        if(g_settings.auto_execute)
        {
            ExecuteSignal(signals[i]);
            MarkSignalProcessed(signals[i].id);
        }
        else
        {
            string msg = "ATLAS Signal: " + signals[i].direction + " " +
                         signals[i].symbol + " @ " +
                         DoubleToString(signals[i].entry_price, 5) +
                         "\nSL: " + DoubleToString(signals[i].stop_price, 5) +
                         " TP: " + DoubleToString(signals[i].target_price, 5) +
                         "\nConfidence: " + IntegerToString(signals[i].confidence) + "%";

            Print("[ATLAS EA] Manual signal: ", msg);
            Alert(msg);
            SendNotification(msg);

            g_http.SendFeedback(signals[i].id, "pending");
            MarkSignalProcessed(signals[i].id);
        }
    }
}

//+------------------------------------------------------------------+
//| Check for closed ATLAS positions (MT4 order history)              |
//+------------------------------------------------------------------+
void CheckClosedPositions()
{
    int total = OrdersHistoryTotal();
    for(int i = total - 1; i >= MathMax(0, total - 10); i--)
    {
        if(!OrderSelect(i, SELECT_BY_POS, MODE_HISTORY))
            continue;

        if(OrderMagicNumber() != InpMagicNumber)
            continue;

        // Check if order closed recently (within 60 seconds)
        if(TimeGMT() - OrderCloseTime() > 60)
            continue;

        string comment = OrderComment();
        int pipe_pos = StringFind(comment, "|");
        if(pipe_pos < 0) continue;

        string signal_id = StringSubstr(comment, pipe_pos + 1);
        if(StringFind(signal_id, SIGNAL_ID_PREFIX) != 0)
            continue;

        double open_price = OrderOpenPrice();
        double close_price = OrderClosePrice();
        double profit = OrderProfit();
        double volume = OrderLots();
        string symbol = OrderSymbol();
        string direction = (OrderType() == OP_BUY) ? "buy" : "sell";

        double pnl_pips = g_risk.CalculatePnlPips(symbol, direction,
                                                    open_price, close_price);

        g_risk.RecordClosedPnl(profit);

        Print("[ATLAS EA] Position closed: ", signal_id, " ", direction,
              " ", DoubleToString(volume, 2), " lots, PnL: ",
              DoubleToString(pnl_pips, 1), " pips");

        g_http.SendFeedback(signal_id, "accepted", open_price, volume,
                            close_price, pnl_pips, 0);
    }
}

//+------------------------------------------------------------------+
//| Expert initialization function                                    |
//+------------------------------------------------------------------+
int OnInit()
{
    Print("[ATLAS EA] Starting ", EA_NAME, " v", EA_VERSION);

    if(StringLen(InpApiKey) == 0 || StringFind(InpApiKey, "ak_") != 0)
    {
        Print("[ATLAS EA] Invalid API key format. Must start with 'ak_'");
        if(InpShowPanel)
        {
            g_panel.Init(InpPanelX, InpPanelY, InpPanelBg, InpPanelText);
            g_panel.Show();
            g_panel.UpdateConnection(false, "Invalid API key");
        }
        return INIT_FAILED;
    }

    g_http.Init(InpApiKey, InpServerUrl);
    g_risk.Init();
    SettingsReset(g_settings);

    Print("[ATLAS EA] Validating API key...");
    if(!g_http.ValidateKey(g_user_plan, g_user_status, g_settings))
    {
        Print("[ATLAS EA] Authentication failed");
        if(InpShowPanel)
        {
            g_panel.Init(InpPanelX, InpPanelY, InpPanelBg, InpPanelText);
            g_panel.Show();
            g_panel.UpdateConnection(false, "Auth failed");
        }
        return INIT_FAILED;
    }

    g_authenticated = true;
    Print("[ATLAS EA] Authenticated. Plan: ", g_user_plan, " Status: ", g_user_status);

    if(g_user_status != "active")
    {
        Print("[ATLAS EA] Account not active (status: ", g_user_status, ")");
        if(InpShowPanel)
        {
            g_panel.Init(InpPanelX, InpPanelY, InpPanelBg, InpPanelText);
            g_panel.Show();
            g_panel.UpdateConnection(false, "Account " + g_user_status);
        }
        return INIT_FAILED;
    }

    if(InpShowPanel)
    {
        g_panel.Init(InpPanelX, InpPanelY, InpPanelBg, InpPanelText);
        g_panel.Show();
        g_panel.UpdateConnection(true, "");
        g_panel.UpdatePlan(g_user_plan);
        g_panel.UpdateMode(g_settings.auto_execute);
    }

    ArrayResize(g_processed_ids, 0);
    g_processed_count = 0;

    if(!EventSetTimer(InpPollInterval))
    {
        Print("[ATLAS EA] Failed to set timer");
        return INIT_FAILED;
    }

    string mode = "MANUAL";
    if(g_settings.auto_execute) mode = "AUTO";
    Print("[ATLAS EA] Ready. Polling every ", InpPollInterval, " seconds. Mode: ", mode);

    return INIT_SUCCEEDED;
}

//+------------------------------------------------------------------+
//| Expert deinitialization function                                  |
//+------------------------------------------------------------------+
void OnDeinit(const int reason)
{
    EventKillTimer();
    g_panel.Destroy();
    Print("[ATLAS EA] Stopped. Reason: ", reason,
          ". Processed ", g_processed_count, " signals.");
}

//+------------------------------------------------------------------+
//| Timer event — poll for signals                                    |
//+------------------------------------------------------------------+
void OnTimer()
{
    if(!g_authenticated)
        return;

    g_risk.CheckDailyReset();

    AtlasSignal signals[];
    int count = 0;
    datetime server_time = 0;

    if(!g_http.FetchActiveSignals(signals, count, server_time))
    {
        g_poll_errors++;
        Print("[ATLAS EA] Poll failed (", g_poll_errors, " consecutive errors)");
        if(InpShowPanel)
            g_panel.UpdateConnection(false, "Error (" + IntegerToString(g_poll_errors) + ")");
        return;
    }

    g_poll_errors = 0;
    g_last_poll = TimeGMT();
    g_signal_count = count;

    if(count > 0)
        ProcessSignals(signals, count);

    // Check for recently closed positions
    CheckClosedPositions();

    if(InpShowPanel)
    {
        g_panel.UpdateConnection(true, "");
        g_panel.UpdateLastPoll(g_last_poll);
        g_panel.UpdateSignals(g_signal_count);

        int open_positions = g_risk.CountOpenPositions(InpMagicNumber);
        g_panel.UpdatePositions(open_positions, g_settings.max_open_trades);

        double equity = AccountEquity();
        double daily_pnl = equity - g_risk.DailyStartBalance();
        g_panel.UpdateDailyPnl(daily_pnl);

        double dd_pct = g_risk.GetDailyDrawdownPct();
        g_panel.UpdateDrawdown(dd_pct, g_settings.max_daily_dd_pct);
    }
}
