//+------------------------------------------------------------------+
//|                                                  ATLAS_Http.mqh  |
//|                                    Copyright 2026, ATLAS FX Ltd. |
//|                                        https://www.atlasfxsignals.com   |
//+------------------------------------------------------------------+
#property copyright "Copyright 2026, ATLAS FX Ltd."
#property link      "https://www.atlasfxsignals.com"
#property strict

#ifndef ATLAS_HTTP_MQH
#define ATLAS_HTTP_MQH

#include "ATLAS_Config.mqh"
#include "ATLAS_Json.mqh"

//+------------------------------------------------------------------+
//| HTTP client for ATLAS API — MT4 version                           |
//| MT4 WebRequest uses cookie/referer signature, not custom headers  |
//| BUT the 6-param version (build 2361+) supports custom headers    |
//+------------------------------------------------------------------+
class AtlasHttp
{
private:
    string m_api_key;
    string m_base_url;
    int    m_timeout;
    int    m_last_status;
    string m_last_error;

    //--- Build headers string
    string BuildHeaders(string content_type = "")
    {
        string hdrs = "";
        hdrs = hdrs + "X-API-Key: " + m_api_key + "\r\n";
        hdrs = hdrs + "User-Agent: " + EA_USER_AGENT + "\r\n";
        hdrs = hdrs + "Accept: application/json\r\n";
        if(content_type != "")
            hdrs = hdrs + "Content-Type: " + content_type + "\r\n";
        return hdrs;
    }

    //--- Execute WebRequest with retry logic
    HttpResponse DoRequest(string method, string url, string body = "")
    {
        HttpResponse response;
        HttpResponseReset(response);

        char  data[];
        char  result[];
        string result_headers;

        // Use StringLen to exclude null terminator from HTTP body
        if(body != "")
            StringToCharArray(body, data, 0, StringLen(body), CP_UTF8);
        else
            ArrayResize(data, 0);

        string content_type = "";
        if(method == "POST" || method == "PUT" || method == "PATCH")
            content_type = "application/json";

        string headers = BuildHeaders(content_type);

        for(int attempt = 0; attempt <= HTTP_MAX_RETRIES; attempt++)
        {
            if(attempt > 0)
            {
                Print("[ATLAS HTTP] Retry ", attempt, " for ", method, " ", url);
                Sleep(HTTP_RETRY_DELAY_MS);
            }

            ResetLastError();

            // MT4 WebRequest with custom headers (build 2361+)
            int status = WebRequest(method, url, headers, m_timeout, data,
                                    result, result_headers);

            if(status == -1)
            {
                int err = GetLastError();
                response.status_code = -1;
                response.error_message = "WebRequest failed (error " +
                    IntegerToString(err) + "). Whitelist " +
                    InpServerUrl + " in Tools > Options > Expert Advisors.";
                Print("[ATLAS HTTP] WebRequest error ", err, " on ", method, " ", url);

                if(attempt < HTTP_MAX_RETRIES) continue;
                return response;
            }

            response.status_code = status;
            response.headers = result_headers;

            if(ArraySize(result) > 0)
                response.body = CharArrayToString(result, 0, WHOLE_ARRAY, CP_UTF8);
            else
                response.body = "";

            if(status >= 200 && status < 300)
            {
                response.success = true;
                m_last_status = status;
                m_last_error = "";
                return response;
            }

            if(status == 429 && attempt < HTTP_MAX_RETRIES)
            {
                Print("[ATLAS HTTP] Rate limited (429), retrying...");
                Sleep(2000);
                continue;
            }

            if(status >= 500 && attempt < HTTP_MAX_RETRIES)
            {
                Print("[ATLAS HTTP] Server error (", status, "), retrying...");
                continue;
            }

            response.success = false;
            response.error_message = "HTTP " + IntegerToString(status) +
                " on " + method + " " + url;

            if(response.body != "")
            {
                JsonParser parser;
                JsonValue *root = parser.Parse(response.body);
                if(root != NULL && root.IsObject())
                {
                    JsonValue *err_obj = root.Get("error");
                    if(err_obj != NULL && err_obj.IsObject())
                        response.error_message = err_obj.GetString("message",
                                                    response.error_message);
                }
                if(root != NULL) delete root;
            }

            m_last_status = status;
            m_last_error = response.error_message;
            Print("[ATLAS HTTP] ", response.error_message);
            return response;
        }

        return response;
    }

public:
    AtlasHttp()
    {
        m_api_key = "";
        m_base_url = "";
        m_timeout = HTTP_TIMEOUT_MS;
        m_last_status = 0;
        m_last_error = "";
    }

    void Init(string api_key, string server_url)
    {
        m_api_key = api_key;
        m_base_url = server_url;

        if(StringGetCharacter(m_base_url, StringLen(m_base_url) - 1) == '/')
            m_base_url = StringSubstr(m_base_url, 0, StringLen(m_base_url) - 1);
        if(StringFind(m_base_url, "/v1") < 0)
            m_base_url = m_base_url + "/v1";
    }

    HttpResponse Get(string endpoint)
    {
        return DoRequest("GET", m_base_url + endpoint);
    }

    HttpResponse Post(string endpoint, string json_body = "{}")
    {
        return DoRequest("POST", m_base_url + endpoint, json_body);
    }

    bool ValidateKey(string &out_plan, string &out_status, AtlasSettings &settings)
    {
        HttpResponse resp = Post("/auth/validate", "{}");
        if(!resp.success)
        {
            Print("[ATLAS HTTP] Auth validation failed: ", resp.error_message);
            return false;
        }

        JsonParser parser;
        JsonValue *root = parser.Parse(resp.body);
        if(root == NULL || root.IsError())
        {
            Print("[ATLAS HTTP] Failed to parse auth response");
            if(root != NULL) delete root;
            return false;
        }

        bool valid = root.GetBool("valid", false);
        if(!valid)
        {
            Print("[ATLAS HTTP] Auth invalid: ", root.GetString("error", "Unknown"));
            delete root;
            return false;
        }

        out_plan = root.GetString("plan", "free");
        out_status = root.GetString("status", "active");

        JsonValue *s = root.Get("settings");
        if(s != NULL && s.IsObject())
        {
            settings.min_confidence         = s.GetInt("min_confidence", InpMinConfidence);
            settings.max_risk_pct           = s.GetDouble("max_risk_pct", InpMaxRiskPercent);
            settings.max_daily_dd_pct       = s.GetDouble("max_daily_dd_pct", InpMaxDailyDD);
            settings.max_weekly_dd_pct      = s.GetDouble("max_weekly_dd_pct", 5.0);
            settings.max_open_trades        = s.GetInt("max_open_trades", InpMaxOpenTrades);
            settings.max_slippage_pips      = s.GetDouble("max_slippage_pips", InpMaxSlippagePips);
            settings.max_entry_deviation_pct = s.GetDouble("max_entry_deviation_pct", 0.5);
            settings.auto_execute           = s.GetBool("auto_execute", InpAutoExecute);

            JsonValue *syms = s.Get("allowed_symbols");
            if(syms != NULL && syms.IsArray())
            {
                int count = syms.Size();
                ArrayResize(settings.allowed_symbols, count);
                for(int i = 0; i < count; i++)
                {
                    JsonValue *sym = syms.At(i);
                    if(sym != NULL && sym.IsString())
                        settings.allowed_symbols[i] = sym.ToStr();
                }
            }
        }

        delete root;
        return true;
    }

    bool FetchActiveSignals(AtlasSignal &signals[], int &count, datetime &server_time)
    {
        count = 0;
        HttpResponse resp = Get("/signals/active");
        if(!resp.success)
        {
            Print("[ATLAS HTTP] Failed to fetch signals: ", resp.error_message);
            return false;
        }

        JsonParser parser;
        JsonValue *root = parser.Parse(resp.body);
        if(root == NULL || root.IsError())
        {
            Print("[ATLAS HTTP] Failed to parse signals response");
            if(root != NULL) delete root;
            return false;
        }

        string st = root.GetString("server_time", "");
        if(st != "")
            server_time = ParseIso8601(st);

        JsonValue *sigs = root.Get("signals");
        if(sigs == NULL || !sigs.IsArray())
        {
            delete root;
            return true;
        }

        count = sigs.Size();
        ArrayResize(signals, count);

        for(int i = 0; i < count; i++)
        {
            JsonValue *sig = sigs.At(i);
            if(sig == NULL || !sig.IsObject()) continue;

            signals[i].id           = sig.GetString("id", "");
            signals[i].symbol       = sig.GetString("symbol", "");
            signals[i].direction    = sig.GetString("direction", "");
            signals[i].confidence   = sig.GetInt("confidence", 0);
            signals[i].regime       = sig.GetString("regime", "");
            signals[i].entry_price  = sig.GetDouble("entry_price", 0);
            signals[i].stop_price   = sig.GetDouble("stop_price", 0);
            signals[i].target_price = sig.GetDouble("target_price", 0);
            signals[i].risk_pips    = sig.GetDouble("risk_pips", 0);
            signals[i].reward_pips  = sig.GetDouble("reward_pips", 0);
            signals[i].rr_ratio     = sig.GetDouble("rr_ratio", 0);
            signals[i].status       = sig.GetString("status", "");
            signals[i].expires_at   = ParseIso8601(sig.GetString("expires_at", ""));
            signals[i].created_at   = ParseIso8601(sig.GetString("created_at", ""));
        }

        delete root;
        return true;
    }

    bool SendFeedback(string signal_id, string action, double executed_price = 0,
                      double executed_size = 0, double close_price = 0,
                      double pnl_pips = 0, double pnl_r = 0,
                      string error_msg = "", string filter_reason = "")
    {
        JsonBuilder jb;
        jb.AddString("signal_id", signal_id);
        jb.AddString("action", action);
        jb.AddString("execution_type", "self_managed");

        if(executed_price > 0)  jb.AddDouble("executed_price", executed_price);
        if(executed_size > 0)   jb.AddDouble("executed_size", executed_size);
        if(close_price > 0)     jb.AddDouble("close_price", close_price);
        if(pnl_pips != 0)       jb.AddDouble("pnl_pips", pnl_pips, 2);
        if(pnl_r != 0)          jb.AddDouble("pnl_r", pnl_r, 2);
        if(error_msg != "")     jb.AddString("error_message", error_msg);
        if(filter_reason != "") jb.AddString("filter_reason", filter_reason);

        HttpResponse resp = Post("/signals/feedback", jb.Build());
        if(!resp.success)
        {
            Print("[ATLAS HTTP] Feedback failed for ", signal_id, ": ", resp.error_message);
            return false;
        }

        return true;
    }

    int    LastStatus()  { return m_last_status; }
    string LastError()   { return m_last_error; }
};

#endif // ATLAS_HTTP_MQH
