//+------------------------------------------------------------------+
//|                                                  ATLAS_Json.mqh  |
//|                                    Copyright 2026, ATLAS FX Ltd. |
//|                                        https://www.atlasfxsignals.com   |
//+------------------------------------------------------------------+
#property copyright "Copyright 2026, ATLAS FX Ltd."
#property link      "https://www.atlasfxsignals.com"
#property strict

#ifndef ATLAS_JSON_MQH
#define ATLAS_JSON_MQH

//+------------------------------------------------------------------+
//| JSON value types                                                  |
//+------------------------------------------------------------------+
enum JSON_TYPE
{
    JSON_NULL,
    JSON_BOOL,
    JSON_INT,
    JSON_DOUBLE,
    JSON_STRING,
    JSON_ARRAY,
    JSON_OBJECT,
    JSON_ERROR
};

//+------------------------------------------------------------------+
//| Forward declaration                                               |
//+------------------------------------------------------------------+
class JsonValue;

//+------------------------------------------------------------------+
//| Key-value pair for JSON objects                                   |
//+------------------------------------------------------------------+
struct JsonPair
{
    string     key;
    JsonValue *value;
};

//+------------------------------------------------------------------+
//| JSON Value — represents any JSON type                             |
//| (MT4-compatible: no interfaces, uses int instead of long where    |
//|  needed, careful with array operations)                           |
//+------------------------------------------------------------------+
class JsonValue
{
private:
    JSON_TYPE   m_type;
    string      m_string_val;
    double      m_double_val;
    int         m_int_val;
    bool        m_bool_val;

    JsonPair    m_pairs[];
    int         m_pair_count;

    JsonValue  *m_items[];
    int         m_item_count;

public:
    JsonValue()
    {
        m_type = JSON_NULL;
        m_string_val = "";
        m_double_val = 0;
        m_int_val = 0;
        m_bool_val = false;
        m_pair_count = 0;
        m_item_count = 0;
    }

    ~JsonValue()
    {
        Clear();
    }

    void Clear()
    {
        for(int i = 0; i < m_pair_count; i++)
        {
            if(m_pairs[i].value != NULL)
            {
                delete m_pairs[i].value;
                m_pairs[i].value = NULL;
            }
        }
        m_pair_count = 0;
        ArrayResize(m_pairs, 0);

        for(int i = 0; i < m_item_count; i++)
        {
            if(m_items[i] != NULL)
            {
                delete m_items[i];
                m_items[i] = NULL;
            }
        }
        m_item_count = 0;
        ArrayResize(m_items, 0);
        m_type = JSON_NULL;
    }

    JSON_TYPE Type()        { return m_type; }
    bool IsNull()           { return m_type == JSON_NULL; }
    bool IsBool()           { return m_type == JSON_BOOL; }
    bool IsInt()            { return m_type == JSON_INT; }
    bool IsDouble()         { return m_type == JSON_DOUBLE; }
    bool IsNumber()         { return m_type == JSON_INT || m_type == JSON_DOUBLE; }
    bool IsString()         { return m_type == JSON_STRING; }
    bool IsArray()          { return m_type == JSON_ARRAY; }
    bool IsObject()         { return m_type == JSON_OBJECT; }
    bool IsError()          { return m_type == JSON_ERROR; }

    string ToStr()          { return m_string_val; }
    double ToDouble()
    {
        if(m_type == JSON_INT) return (double)m_int_val;
        return m_double_val;
    }
    int    ToInt()
    {
        if(m_type == JSON_DOUBLE) return (int)m_double_val;
        return m_int_val;
    }
    bool   ToBool()         { return m_bool_val; }

    int Size()
    {
        if(m_type == JSON_ARRAY) return m_item_count;
        if(m_type == JSON_OBJECT) return m_pair_count;
        return 0;
    }

    JsonValue *Get(string key)
    {
        if(m_type != JSON_OBJECT)
            return NULL;
        for(int i = 0; i < m_pair_count; i++)
        {
            if(m_pairs[i].key == key)
                return m_pairs[i].value;
        }
        return NULL;
    }

    bool Has(string key)
    {
        return Get(key) != NULL;
    }

    string GetString(string key, string default_val = "")
    {
        JsonValue *v = Get(key);
        if(v == NULL || v.IsNull()) return default_val;
        return v.ToStr();
    }

    int GetInt(string key, int default_val = 0)
    {
        JsonValue *v = Get(key);
        if(v == NULL || v.IsNull()) return default_val;
        return v.ToInt();
    }

    double GetDouble(string key, double default_val = 0.0)
    {
        JsonValue *v = Get(key);
        if(v == NULL || v.IsNull()) return default_val;
        return v.ToDouble();
    }

    bool GetBool(string key, bool default_val = false)
    {
        JsonValue *v = Get(key);
        if(v == NULL || v.IsNull()) return default_val;
        return v.ToBool();
    }

    JsonValue *At(int index)
    {
        if(m_type != JSON_ARRAY || index < 0 || index >= m_item_count)
            return NULL;
        return m_items[index];
    }

    void SetNull()           { m_type = JSON_NULL; }
    void SetError(string msg){ m_type = JSON_ERROR; m_string_val = msg; }
    void SetBool(bool v)     { m_type = JSON_BOOL; m_bool_val = v; }
    void SetInt(int v)       { m_type = JSON_INT; m_int_val = v; }
    void SetDouble(double v) { m_type = JSON_DOUBLE; m_double_val = v; }
    void SetString(string v) { m_type = JSON_STRING; m_string_val = v; }

    void SetArray()  { m_type = JSON_ARRAY; m_item_count = 0; }
    void SetObject() { m_type = JSON_OBJECT; m_pair_count = 0; }

    void AddItem(JsonValue *item)
    {
        int new_size = m_item_count + 1;
        ArrayResize(m_items, new_size);
        m_items[m_item_count] = item;
        m_item_count = new_size;
    }

    void AddPair(string key, JsonValue *value)
    {
        int new_size = m_pair_count + 1;
        ArrayResize(m_pairs, new_size);
        m_pairs[m_pair_count].key = key;
        m_pairs[m_pair_count].value = value;
        m_pair_count = new_size;
    }
};

//+------------------------------------------------------------------+
//| JSON Parser (MT4 compatible)                                      |
//+------------------------------------------------------------------+
class JsonParser
{
private:
    string m_json;
    int    m_pos;
    int    m_len;

    void SkipWhitespace()
    {
        while(m_pos < m_len)
        {
            ushort ch = StringGetCharacter(m_json, m_pos);
            if(ch == ' ' || ch == '\t' || ch == '\r' || ch == '\n')
                m_pos++;
            else
                break;
        }
    }

    ushort Peek()
    {
        if(m_pos >= m_len) return 0;
        return StringGetCharacter(m_json, m_pos);
    }

    ushort Next()
    {
        if(m_pos >= m_len) return 0;
        ushort ch = StringGetCharacter(m_json, m_pos);
        m_pos++;
        return ch;
    }

    string ParseString()
    {
        string result = "";
        while(m_pos < m_len)
        {
            ushort ch = Next();
            if(ch == '"')
                return result;
            if(ch == '\\')
            {
                ushort esc = Next();
                if(esc == '"')       result = result + "\"";
                else if(esc == '\\') result = result + "\\";
                else if(esc == '/')  result = result + "/";
                else if(esc == 'n')  result = result + "\n";
                else if(esc == 'r')  result = result + "\r";
                else if(esc == 't')  result = result + "\t";
                else if(esc == 'u')
                {
                    // Skip 4 hex digits, insert '?'
                    for(int i = 0; i < 4 && m_pos < m_len; i++) m_pos++;
                    result = result + "?";
                }
                else result = result + ShortToString(esc);
            }
            else
            {
                result = result + ShortToString(ch);
            }
        }
        return result;
    }

    JsonValue *ParseNumber()
    {
        int start = m_pos;
        bool is_float = false;

        if(Peek() == '-') m_pos++;

        while(m_pos < m_len)
        {
            ushort ch = Peek();
            if(ch >= '0' && ch <= '9') m_pos++;
            else break;
        }

        if(m_pos < m_len && Peek() == '.')
        {
            is_float = true;
            m_pos++;
            while(m_pos < m_len)
            {
                ushort ch = Peek();
                if(ch >= '0' && ch <= '9') m_pos++;
                else break;
            }
        }

        if(m_pos < m_len && (Peek() == 'e' || Peek() == 'E'))
        {
            is_float = true;
            m_pos++;
            if(m_pos < m_len && (Peek() == '+' || Peek() == '-')) m_pos++;
            while(m_pos < m_len)
            {
                ushort ch = Peek();
                if(ch >= '0' && ch <= '9') m_pos++;
                else break;
            }
        }

        string num_str = StringSubstr(m_json, start, m_pos - start);
        JsonValue *val = new JsonValue();

        if(is_float)
            val.SetDouble(StringToDouble(num_str));
        else
            val.SetInt((int)StringToInteger(num_str));

        return val;
    }

    JsonValue *ParseValue()
    {
        SkipWhitespace();

        if(m_pos >= m_len)
        {
            JsonValue *val = new JsonValue();
            val.SetError("Unexpected end of input");
            return val;
        }

        ushort ch = Peek();

        if(ch == '"')
        {
            m_pos++;
            JsonValue *val = new JsonValue();
            val.SetString(ParseString());
            return val;
        }

        if(ch == '{')
        {
            m_pos++;
            JsonValue *obj = new JsonValue();
            obj.SetObject();
            SkipWhitespace();
            if(Peek() == '}') { m_pos++; return obj; }

            while(m_pos < m_len)
            {
                SkipWhitespace();
                if(Peek() != '"') { obj.SetError("Expected key"); return obj; }
                m_pos++;
                string key = ParseString();
                SkipWhitespace();
                if(Next() != ':') { obj.SetError("Expected ':'"); return obj; }
                JsonValue *val = ParseValue();
                obj.AddPair(key, val);
                SkipWhitespace();
                ushort sep = Peek();
                if(sep == ',') { m_pos++; continue; }
                else if(sep == '}') { m_pos++; break; }
                else { obj.SetError("Expected ',' or '}'"); return obj; }
            }
            return obj;
        }

        if(ch == '[')
        {
            m_pos++;
            JsonValue *arr = new JsonValue();
            arr.SetArray();
            SkipWhitespace();
            if(Peek() == ']') { m_pos++; return arr; }

            while(m_pos < m_len)
            {
                JsonValue *item = ParseValue();
                arr.AddItem(item);
                SkipWhitespace();
                ushort sep = Peek();
                if(sep == ',') { m_pos++; continue; }
                else if(sep == ']') { m_pos++; break; }
                else { arr.SetError("Expected ',' or ']'"); return arr; }
            }
            return arr;
        }

        if(ch == '-' || (ch >= '0' && ch <= '9'))
            return ParseNumber();

        if(ch == 't' && StringSubstr(m_json, m_pos, 4) == "true")
        {
            m_pos += 4;
            JsonValue *val = new JsonValue();
            val.SetBool(true);
            return val;
        }

        if(ch == 'f' && StringSubstr(m_json, m_pos, 5) == "false")
        {
            m_pos += 5;
            JsonValue *val = new JsonValue();
            val.SetBool(false);
            return val;
        }

        if(ch == 'n' && StringSubstr(m_json, m_pos, 4) == "null")
        {
            m_pos += 4;
            JsonValue *val = new JsonValue();
            val.SetNull();
            return val;
        }

        JsonValue *val = new JsonValue();
        val.SetError("Unexpected character");
        return val;
    }

public:
    JsonValue *Parse(string json_str)
    {
        m_json = json_str;
        m_pos = 0;
        m_len = StringLen(json_str);

        if(m_len == 0)
        {
            JsonValue *val = new JsonValue();
            val.SetError("Empty input");
            return val;
        }

        return ParseValue();
    }
};

//+------------------------------------------------------------------+
//| JSON Builder (MT4 compatible)                                     |
//+------------------------------------------------------------------+
class JsonBuilder
{
private:
    string m_buffer;
    bool   m_first;

public:
    JsonBuilder()
    {
        m_buffer = "{";
        m_first = true;
    }

    void AddString(string key, string value)
    {
        if(!m_first) m_buffer = m_buffer + ",";
        m_first = false;
        m_buffer = m_buffer + "\"" + key + "\":\"" + value + "\"";
    }

    void AddInt(string key, int value)
    {
        if(!m_first) m_buffer = m_buffer + ",";
        m_first = false;
        m_buffer = m_buffer + "\"" + key + "\":" + IntegerToString(value);
    }

    void AddDouble(string key, double value, int digits = 5)
    {
        if(!m_first) m_buffer = m_buffer + ",";
        m_first = false;
        m_buffer = m_buffer + "\"" + key + "\":" + DoubleToString(value, digits);
    }

    void AddBool(string key, bool value)
    {
        if(!m_first) m_buffer = m_buffer + ",";
        m_first = false;
        string v = "false";
        if(value) v = "true";
        m_buffer = m_buffer + "\"" + key + "\":" + v;
    }

    void AddNull(string key)
    {
        if(!m_first) m_buffer = m_buffer + ",";
        m_first = false;
        m_buffer = m_buffer + "\"" + key + "\":null";
    }

    string Build()
    {
        return m_buffer + "}";
    }
};

#endif // ATLAS_JSON_MQH
