//+------------------------------------------------------------------+
//|                                                 ATLAS_Panel.mqh  |
//|                                    Copyright 2026, ATLAS FX Ltd. |
//|                                        https://www.atlasfxsignals.com   |
//+------------------------------------------------------------------+
#property copyright "Copyright 2026, ATLAS FX Ltd."
#property link      "https://www.atlasfxsignals.com"
#property strict

#ifndef ATLAS_PANEL_MQH
#define ATLAS_PANEL_MQH

#include "ATLAS_Config.mqh"

#define PANEL_PREFIX "ATLAS_"

enum PANEL_ROW
{
    ROW_TITLE,
    ROW_SEPARATOR1,
    ROW_CONNECTION,
    ROW_PLAN,
    ROW_LAST_POLL,
    ROW_SEPARATOR2,
    ROW_SIGNALS,
    ROW_POSITIONS,
    ROW_DAILY_PNL,
    ROW_DRAWDOWN,
    ROW_SEPARATOR3,
    ROW_MODE,
    ROW_VERSION,
    ROW_COUNT
};

//+------------------------------------------------------------------+
//| On-chart status panel — MT4 version                               |
//+------------------------------------------------------------------+
class AtlasPanel
{
private:
    bool   m_visible;
    int    m_x;
    int    m_y;
    int    m_row_height;
    color  m_bg_color;
    color  m_text_color;
    int    m_font_size;
    string m_font_name;
    int    m_panel_width;

    void CreateLabel(string name, int x, int y, string text,
                     color clr, int font_size = 0)
    {
        if(font_size == 0) font_size = m_font_size;

        if(ObjectFind(name) >= 0)
            ObjectDelete(name);

        ObjectCreate(name, OBJ_LABEL, 0, 0, 0);
        ObjectSet(name, OBJPROP_XDISTANCE, x);
        ObjectSet(name, OBJPROP_YDISTANCE, y);
        ObjectSet(name, OBJPROP_CORNER, CORNER_LEFT_UPPER);
        ObjectSetText(name, text, font_size, m_font_name, clr);
        ObjectSet(name, OBJPROP_SELECTABLE, false);
    }

    void CreateBackground()
    {
        string name = PANEL_PREFIX + "BG";
        if(ObjectFind(name) >= 0)
            ObjectDelete(name);

        ObjectCreate(name, OBJ_RECTANGLE_LABEL, 0, 0, 0);
        ObjectSet(name, OBJPROP_XDISTANCE, m_x - 5);
        ObjectSet(name, OBJPROP_YDISTANCE, m_y - 5);
        ObjectSet(name, OBJPROP_XSIZE, m_panel_width);
        ObjectSet(name, OBJPROP_YSIZE, ROW_COUNT * m_row_height + 10);
        ObjectSet(name, OBJPROP_BGCOLOR, m_bg_color);
        ObjectSet(name, OBJPROP_BORDER_TYPE, BORDER_FLAT);
        ObjectSet(name, OBJPROP_BORDER_COLOR, clrDimGray);
        ObjectSet(name, OBJPROP_CORNER, CORNER_LEFT_UPPER);
        ObjectSet(name, OBJPROP_SELECTABLE, false);
        ObjectSet(name, OBJPROP_BACK, false);
    }

    string RowName(int row)
    {
        return PANEL_PREFIX + "R" + IntegerToString(row);
    }

    int RowY(int row)
    {
        return m_y + row * m_row_height;
    }

public:
    AtlasPanel()
    {
        m_visible = false;
        m_x = 10;
        m_y = 30;
        m_row_height = 18;
        m_bg_color = clrMidnightBlue;
        m_text_color = clrWhite;
        m_font_size = 9;
        m_font_name = "Consolas";
        m_panel_width = 260;
    }

    void Init(int x, int y, color bg, color text_clr)
    {
        m_x = x;
        m_y = y;
        m_bg_color = bg;
        m_text_color = text_clr;
    }

    void Show()
    {
        m_visible = true;
        CreateBackground();

        CreateLabel(RowName(ROW_TITLE), m_x, RowY(ROW_TITLE),
                    "  ATLAS FX v" + EA_VERSION, clrGold, 10);

        CreateLabel(RowName(ROW_SEPARATOR1), m_x, RowY(ROW_SEPARATOR1),
                    " -------------------------", clrDimGray);
        CreateLabel(RowName(ROW_SEPARATOR2), m_x, RowY(ROW_SEPARATOR2),
                    " -------------------------", clrDimGray);
        CreateLabel(RowName(ROW_SEPARATOR3), m_x, RowY(ROW_SEPARATOR3),
                    " -------------------------", clrDimGray);

        UpdateConnection(false, "");
        UpdatePlan("---");
        UpdateLastPoll(0);
        UpdateSignals(0);
        UpdatePositions(0, 0);
        UpdateDailyPnl(0);
        UpdateDrawdown(0, 0);
        UpdateMode(false);
        UpdateVersion();
    }

    void Destroy()
    {
        ObjectDelete(PANEL_PREFIX + "BG");
        for(int i = 0; i < ROW_COUNT; i++)
            ObjectDelete(RowName(i));
        m_visible = false;
    }

    void UpdateConnection(bool connected, string message)
    {
        color clr = connected ? clrLime : clrRed;
        string status = connected ? "Connected" : "Disconnected";
        if(message != "") status = message;
        CreateLabel(RowName(ROW_CONNECTION), m_x, RowY(ROW_CONNECTION),
                    " Status:   " + status, clr);
    }

    void UpdatePlan(string plan)
    {
        CreateLabel(RowName(ROW_PLAN), m_x, RowY(ROW_PLAN),
                    " Plan:     " + plan, m_text_color);
    }

    void UpdateLastPoll(datetime last_poll)
    {
        string poll_str = "Never";
        if(last_poll > 0)
            poll_str = TimeToStr(last_poll, TIME_MINUTES | TIME_SECONDS);
        CreateLabel(RowName(ROW_LAST_POLL), m_x, RowY(ROW_LAST_POLL),
                    " Last Poll:" + poll_str, m_text_color);
    }

    void UpdateSignals(int signal_count)
    {
        CreateLabel(RowName(ROW_SIGNALS), m_x, RowY(ROW_SIGNALS),
                    " Signals:  " + IntegerToString(signal_count) + " active",
                    m_text_color);
    }

    void UpdatePositions(int open_count, int max_count)
    {
        string text = " Trades:   " + IntegerToString(open_count) +
                      " / " + IntegerToString(max_count);
        color clr = (open_count >= max_count) ? clrOrange : m_text_color;
        CreateLabel(RowName(ROW_POSITIONS), m_x, RowY(ROW_POSITIONS), text, clr);
    }

    void UpdateDailyPnl(double pnl)
    {
        string sign = pnl >= 0 ? "+" : "";
        string text = " Daily PnL:" + sign + DoubleToString(pnl, 2);
        color clr = pnl >= 0 ? clrLime : clrRed;
        CreateLabel(RowName(ROW_DAILY_PNL), m_x, RowY(ROW_DAILY_PNL), text, clr);
    }

    void UpdateDrawdown(double dd_pct, double max_dd_pct)
    {
        string text = " Drawdown: " + DoubleToString(dd_pct, 1) + "% / " +
                      DoubleToString(max_dd_pct, 1) + "%";
        color clr = m_text_color;
        if(dd_pct > max_dd_pct * 0.8) clr = clrOrange;
        if(dd_pct >= max_dd_pct)       clr = clrRed;
        CreateLabel(RowName(ROW_DRAWDOWN), m_x, RowY(ROW_DRAWDOWN), text, clr);
    }

    void UpdateMode(bool auto_execute)
    {
        string mode = "MANUAL";
        if(auto_execute) mode = "AUTO";
        color clr = auto_execute ? clrLime : clrYellow;
        CreateLabel(RowName(ROW_MODE), m_x, RowY(ROW_MODE),
                    " Mode:     " + mode, clr);
    }

    void UpdateVersion()
    {
        CreateLabel(RowName(ROW_VERSION), m_x, RowY(ROW_VERSION),
                    " Magic:    " + IntegerToString(InpMagicNumber), clrDimGray);
    }

    bool IsVisible() { return m_visible; }
};

#endif // ATLAS_PANEL_MQH
