//+------------------------------------------------------------------+
//|                                                  ATLAS_Risk.mqh  |
//|                                    Copyright 2026, ATLAS FX Ltd. |
//|                                        https://www.atlasfxsignals.com   |
//+------------------------------------------------------------------+
#property copyright "Copyright 2026, ATLAS FX Ltd."
#property link      "https://www.atlasfxsignals.com"
#property strict

#ifndef ATLAS_RISK_MQH
#define ATLAS_RISK_MQH

#include "ATLAS_Config.mqh"

//+------------------------------------------------------------------+
//| Risk management — MT4 version                                     |
//| Uses MT4 functions: AccountBalance(), MarketInfo(), OrdersTotal() |
//+------------------------------------------------------------------+
class AtlasRisk
{
private:
    double m_daily_start_balance;
    double m_daily_realized_pnl;
    int    m_daily_reset_day;
    int    m_open_atlas_positions;

public:
    AtlasRisk()
    {
        m_daily_start_balance = 0;
        m_daily_realized_pnl = 0;
        m_daily_reset_day = 0;
        m_open_atlas_positions = 0;
    }

    void Init()
    {
        m_daily_start_balance = AccountBalance();
        m_daily_realized_pnl = 0;
        m_daily_reset_day = TimeDayOfYear(TimeGMT());
    }

    void CheckDailyReset()
    {
        int today = TimeDayOfYear(TimeGMT());
        if(today != m_daily_reset_day)
        {
            m_daily_start_balance = AccountBalance();
            m_daily_realized_pnl = 0;
            m_daily_reset_day = today;
            Print("[ATLAS RISK] Daily reset. Start balance: ",
                  DoubleToString(m_daily_start_balance, 2));
        }
    }

    void RecordClosedPnl(double pnl)
    {
        m_daily_realized_pnl += pnl;
    }

    //+------------------------------------------------------------------+
    //| Calculate position size in lots                                   |
    //+------------------------------------------------------------------+
    double CalculateLotSize(string symbol, double entry_price, double stop_price,
                            double risk_pct)
    {
        if(entry_price <= 0 || stop_price <= 0 || risk_pct <= 0)
            return 0;

        double balance = AccountBalance();
        if(balance <= 0) return 0;

        double risk_amount = balance * risk_pct / 100.0;

        double stop_distance = MathAbs(entry_price - stop_price);
        if(stop_distance == 0) return 0;

        // MT4: MarketInfo for tick value and tick size
        double tick_value = MarketInfo(symbol, MODE_TICKVALUE);
        double tick_size = MarketInfo(symbol, MODE_TICKSIZE);

        if(tick_value <= 0 || tick_size <= 0)
        {
            Print("[ATLAS RISK] Invalid tick info for ", symbol);
            return 0;
        }

        double cost_per_lot = (stop_distance / tick_size) * tick_value;
        if(cost_per_lot <= 0) return 0;

        double lots = risk_amount / cost_per_lot;

        return NormalizeLots(symbol, lots);
    }

    //+------------------------------------------------------------------+
    //| Normalize lot size to broker constraints (MT4)                    |
    //+------------------------------------------------------------------+
    double NormalizeLots(string symbol, double lots)
    {
        double min_lot = MarketInfo(symbol, MODE_MINLOT);
        double max_lot = MarketInfo(symbol, MODE_MAXLOT);
        double step = MarketInfo(symbol, MODE_LOTSTEP);

        if(step <= 0) step = 0.01;
        if(min_lot <= 0) min_lot = 0.01;
        if(max_lot <= 0) max_lot = 100.0;

        lots = MathFloor(lots / step) * step;

        if(lots < min_lot) lots = 0;
        if(lots > max_lot) lots = max_lot;

        int digits = (int)MathCeil(-MathLog10(step));
        lots = NormalizeDouble(lots, digits);

        return lots;
    }

    //+------------------------------------------------------------------+
    //| Adjust SL/TP to preserve R:R from actual fill price              |
    //+------------------------------------------------------------------+
    void AdjustSlTp(double signal_entry, double actual_entry,
                    double signal_sl, double signal_tp,
                    double &adjusted_sl, double &adjusted_tp)
    {
        double slippage = actual_entry - signal_entry;
        adjusted_sl = signal_sl + slippage;
        adjusted_tp = signal_tp + slippage;
    }

    //+------------------------------------------------------------------+
    //| Check slippage acceptability                                      |
    //+------------------------------------------------------------------+
    bool IsSlippageAcceptable(string symbol, double signal_entry, double current_price,
                              double max_slippage_pips)
    {
        double point = MarketInfo(symbol, MODE_POINT);
        int digits = (int)MarketInfo(symbol, MODE_DIGITS);

        if(point <= 0) return false;

        double pip_size = point;
        if(digits == 5 || digits == 3)
            pip_size = point * 10;

        double deviation_pips = MathAbs(current_price - signal_entry) / pip_size;
        return deviation_pips <= max_slippage_pips;
    }

    //+------------------------------------------------------------------+
    //| Daily drawdown percentage                                         |
    //+------------------------------------------------------------------+
    double GetDailyDrawdownPct()
    {
        CheckDailyReset();
        if(m_daily_start_balance <= 0) return 0;

        double equity = AccountEquity();
        double current_pnl = equity - m_daily_start_balance;

        if(current_pnl >= 0) return 0;
        return MathAbs(current_pnl) / m_daily_start_balance * 100.0;
    }

    bool IsDailyDrawdownBreached(double max_dd_pct)
    {
        return GetDailyDrawdownPct() >= max_dd_pct;
    }

    //+------------------------------------------------------------------+
    //| Count open positions by magic number (MT4 order pool)            |
    //+------------------------------------------------------------------+
    int CountOpenPositions(int magic_number)
    {
        int count = 0;
        int total = OrdersTotal();
        for(int i = 0; i < total; i++)
        {
            if(!OrderSelect(i, SELECT_BY_POS, MODE_TRADES))
                continue;
            if(OrderMagicNumber() == magic_number &&
               (OrderType() == OP_BUY || OrderType() == OP_SELL))
                count++;
        }
        m_open_atlas_positions = count;
        return count;
    }

    bool IsSymbolAllowed(string signal_symbol, string &allowed[])
    {
        int count = ArraySize(allowed);
        if(count == 0) return true;

        for(int i = 0; i < count; i++)
        {
            if(allowed[i] == signal_symbol)
                return true;
        }
        return false;
    }

    bool IsSignalExpired(datetime expires_at)
    {
        return TimeGMT() >= expires_at;
    }

    bool IsMarketOpen(string symbol)
    {
        // MT4: check if we can get current price
        double bid = MarketInfo(symbol, MODE_BID);
        return bid > 0;
    }

    double CalculatePnlPips(string symbol, string direction,
                            double open_price, double close_price)
    {
        double point = MarketInfo(symbol, MODE_POINT);
        int digits = (int)MarketInfo(symbol, MODE_DIGITS);

        if(point <= 0) return 0;

        double pip_size = point;
        if(digits == 5 || digits == 3)
            pip_size = point * 10;

        double diff = close_price - open_price;
        if(direction == "sell")
            diff = -diff;

        return diff / pip_size;
    }

    double CalculatePnlR(double open_price, double close_price, double stop_price)
    {
        double risk = MathAbs(open_price - stop_price);
        if(risk == 0) return 0;
        return (close_price - open_price) / risk;
    }

    double DailyStartBalance()  { return m_daily_start_balance; }
    double DailyRealizedPnl()   { return m_daily_realized_pnl; }
    int    OpenPositions()      { return m_open_atlas_positions; }
};

#endif // ATLAS_RISK_MQH
