//+------------------------------------------------------------------+
//|                                                 ATLAS_Config.mqh |
//|                                    Copyright 2026, ATLAS FX Ltd. |
//|                                        https://www.atlasfxsignals.com   |
//+------------------------------------------------------------------+
#property copyright "Copyright 2026, ATLAS FX Ltd."
#property link      "https://www.atlasfxsignals.com"
#property strict

#ifndef ATLAS_CONFIG_MQH
#define ATLAS_CONFIG_MQH

//+------------------------------------------------------------------+
//| EA version                                                        |
//+------------------------------------------------------------------+
#define EA_VERSION        "1.0.0"
#define EA_NAME           "ATLAS EA"
#define EA_USER_AGENT     "ATLAS-EA/1.0 (MT5)"

//+------------------------------------------------------------------+
//| Input parameters — user-configurable in EA settings               |
//+------------------------------------------------------------------+
input group "=== Connection ==="
input string   InpApiKey          = "";                              // API Key (ak_...)
input string   InpServerUrl       = "https://api.atlasfxsignals.com";      // API Server URL

input group "=== Risk Management ==="
input int      InpMinConfidence   = 60;                              // Minimum signal confidence (0-100)
input double   InpMaxRiskPercent  = 1.0;                             // Risk per trade (% of balance)
input double   InpMaxDailyDD     = 3.0;                              // Max daily drawdown (%)
input int      InpMaxOpenTrades   = 3;                               // Max simultaneous positions
input double   InpMaxSlippagePips = 1000.0;                           // Max entry price deviation (pips)

input group "=== Execution ==="
input bool     InpAutoExecute     = false;                           // Auto-execute signals
input int      InpMagicNumber     = 20260224;                        // EA magic number
input int      InpPollInterval    = 10;                              // Polling interval (seconds)
input string   InpSymbolSuffix    = "";                              // Broker symbol suffix (e.g., ".m")

input group "=== Display ==="
input bool     InpShowPanel       = true;                            // Show status panel on chart
input int      InpPanelX          = 10;                              // Panel X position
input int      InpPanelY          = 30;                              // Panel Y position
input color    InpPanelBg         = clrMidnightBlue;                 // Panel background color
input color    InpPanelText       = clrWhite;                        // Panel text color

//+------------------------------------------------------------------+
//| HTTP configuration constants                                      |
//+------------------------------------------------------------------+
#define HTTP_TIMEOUT_MS     10000    // 10 second timeout for WebRequest
#define HTTP_MAX_RETRIES    1        // Retry once on failure
#define HTTP_RETRY_DELAY_MS 1000     // 1 second between retries

//+------------------------------------------------------------------+
//| Signal processing constants                                       |
//+------------------------------------------------------------------+
#define MAX_PROCESSED_SIGNALS 500    // Max tracked signal IDs
#define SIGNAL_ID_PREFIX      "sig_" // Signal ID prefix

//+------------------------------------------------------------------+
//| Settings struct — populated from auth/validate response           |
//+------------------------------------------------------------------+
struct AtlasSettings
{
    int    min_confidence;
    double max_risk_pct;
    double max_daily_dd_pct;
    double max_weekly_dd_pct;
    int    max_open_trades;
    string allowed_symbols[];
    double max_slippage_pips;
    double max_entry_deviation_pct;
    bool   auto_execute;

    void Reset()
    {
        min_confidence         = InpMinConfidence;
        max_risk_pct           = InpMaxRiskPercent;
        max_daily_dd_pct       = InpMaxDailyDD;
        max_weekly_dd_pct      = 5.0;
        max_open_trades        = InpMaxOpenTrades;
        ArrayResize(allowed_symbols, 1);
        allowed_symbols[0]     = "XAUUSD";
        max_slippage_pips      = InpMaxSlippagePips;
        max_entry_deviation_pct = 0.5;
        auto_execute           = InpAutoExecute;
    }
};

//+------------------------------------------------------------------+
//| Signal data struct — parsed from JSON response                    |
//+------------------------------------------------------------------+
struct AtlasSignal
{
    string   id;
    string   symbol;
    string   direction;      // "buy" or "sell"
    int      confidence;
    string   regime;
    double   entry_price;
    double   stop_price;
    double   target_price;
    double   risk_pips;
    double   reward_pips;
    double   rr_ratio;
    string   status;
    datetime expires_at;
    datetime created_at;
};

//+------------------------------------------------------------------+
//| HTTP response struct                                              |
//+------------------------------------------------------------------+
struct HttpResponse
{
    int    status_code;
    string body;
    string headers;
    bool   success;
    string error_message;

    void Reset()
    {
        status_code   = 0;
        body          = "";
        headers       = "";
        success       = false;
        error_message = "";
    }
};

//+------------------------------------------------------------------+
//| Helper: Build API endpoint URL                                    |
//+------------------------------------------------------------------+
string BuildApiUrl(string endpoint)
{
    string base = InpServerUrl;
    // Remove trailing slash if present
    if(StringGetCharacter(base, StringLen(base) - 1) == '/')
        base = StringSubstr(base, 0, StringLen(base) - 1);

    // Ensure /v1 prefix
    if(StringFind(base, "/v1") < 0)
        base += "/v1";

    return base + endpoint;
}

//+------------------------------------------------------------------+
//| Helper: Build broker symbol from signal symbol                    |
//+------------------------------------------------------------------+
string BuildBrokerSymbol(string signal_symbol)
{
    return signal_symbol + InpSymbolSuffix;
}

//+------------------------------------------------------------------+
//| Helper: Parse ISO 8601 datetime string to MQL5 datetime          |
//| Handles: "2026-02-24T12:00:00Z" and "2026-02-24T12:00:00+00:00" |
//+------------------------------------------------------------------+
datetime ParseIso8601(string iso_str)
{
    if(StringLen(iso_str) < 19)
        return 0;

    // Extract: YYYY-MM-DDTHH:MM:SS
    string date_part = StringSubstr(iso_str, 0, 10);  // "2026-02-24"
    string time_part = StringSubstr(iso_str, 11, 8);   // "12:00:00"

    // Replace dashes with dots for MQL5 StringToTime: "2026.02.24"
    StringReplace(date_part, "-", ".");

    string datetime_str = date_part + " " + time_part;
    return StringToTime(datetime_str);
}

//+------------------------------------------------------------------+
//| Helper: Format datetime to ISO 8601 string                       |
//+------------------------------------------------------------------+
string FormatIso8601(datetime dt)
{
    MqlDateTime mdt;
    TimeToStruct(dt, mdt);
    return StringFormat("%04d-%02d-%02dT%02d:%02d:%02dZ",
                        mdt.year, mdt.mon, mdt.day,
                        mdt.hour, mdt.min, mdt.sec);
}

#endif // ATLAS_CONFIG_MQH
