//+------------------------------------------------------------------+
//|                                                    ATLAS_EA.mq5  |
//|                                    Copyright 2026, ATLAS FX Ltd. |
//|                                        https://www.atlasfxsignals.com   |
//+------------------------------------------------------------------+
#property copyright "Copyright 2026, ATLAS FX Ltd."
#property link      "https://www.atlasfxsignals.com"
#property version   "1.00"
#property description "ATLAS FX Expert Advisor — receives H4 trading signals via API"
#property strict

//+------------------------------------------------------------------+
//| Includes                                                          |
//+------------------------------------------------------------------+
#include "ATLAS_Config.mqh"
#include "ATLAS_Json.mqh"
#include "ATLAS_Http.mqh"
#include "ATLAS_Risk.mqh"
#include "ATLAS_Panel.mqh"

//+------------------------------------------------------------------+
//| Globals                                                           |
//+------------------------------------------------------------------+
AtlasHttp   g_http;
AtlasRisk   g_risk;
AtlasPanel  g_panel;
AtlasSettings g_settings;

// Processed signal IDs — prevents duplicate execution
string g_processed_ids[];
int    g_processed_count = 0;

// Session state
bool     g_authenticated = false;
string   g_user_plan = "";
string   g_user_status = "";
datetime g_last_poll = 0;
int      g_signal_count = 0;
int      g_poll_errors = 0;

//+------------------------------------------------------------------+
//| Check if a signal ID has already been processed                   |
//+------------------------------------------------------------------+
bool IsSignalProcessed(string signal_id)
{
    for(int i = 0; i < g_processed_count; i++)
    {
        if(g_processed_ids[i] == signal_id)
            return true;
    }
    return false;
}

//+------------------------------------------------------------------+
//| Mark a signal ID as processed                                     |
//+------------------------------------------------------------------+
void MarkSignalProcessed(string signal_id)
{
    if(g_processed_count >= MAX_PROCESSED_SIGNALS)
    {
        // Evict oldest half when full
        int keep = MAX_PROCESSED_SIGNALS / 2;
        for(int i = 0; i < keep; i++)
            g_processed_ids[i] = g_processed_ids[g_processed_count - keep + i];
        g_processed_count = keep;
    }

    ArrayResize(g_processed_ids, g_processed_count + 1);
    g_processed_ids[g_processed_count] = signal_id;
    g_processed_count++;
}

//+------------------------------------------------------------------+
//| Execute a trade for a signal                                      |
//+------------------------------------------------------------------+
bool ExecuteSignal(const AtlasSignal &signal)
{
    string broker_symbol = BuildBrokerSymbol(signal.symbol);

    // Verify symbol exists
    if(!SymbolInfoInteger(broker_symbol, SYMBOL_EXIST))
    {
        PrintFormat("[ATLAS EA] Symbol %s not found (broker symbol: %s)",
                    signal.symbol, broker_symbol);
        g_http.SendFeedback(signal.id, "error", 0, 0, 0, 0, 0,
                            "Symbol not found: " + broker_symbol);
        return false;
    }

    // Select symbol in market watch
    SymbolSelect(broker_symbol, true);

    // Get current price
    double ask = SymbolInfoDouble(broker_symbol, SYMBOL_ASK);
    double bid = SymbolInfoDouble(broker_symbol, SYMBOL_BID);
    int digits = (int)SymbolInfoInteger(broker_symbol, SYMBOL_DIGITS);

    double current_price = (signal.direction == "buy") ? ask : bid;

    // Check slippage
    if(!g_risk.IsSlippageAcceptable(broker_symbol, signal.entry_price,
                                     current_price, g_settings.max_slippage_pips))
    {
        PrintFormat("[ATLAS EA] Slippage too high for %s: entry=%.5f current=%.5f",
                    signal.id, signal.entry_price, current_price);
        g_http.SendFeedback(signal.id, "filtered", 0, 0, 0, 0, 0,
                            "", "slippage_exceeded");
        return false;
    }

    // Calculate lot size
    double lots = g_risk.CalculateLotSize(broker_symbol, current_price,
                                           signal.stop_price,
                                           g_settings.max_risk_pct);
    if(lots <= 0)
    {
        PrintFormat("[ATLAS EA] Cannot calculate lot size for %s", signal.id);
        g_http.SendFeedback(signal.id, "error", 0, 0, 0, 0, 0,
                            "Position size too small");
        return false;
    }

    // Adjust SL/TP for actual entry price (preserve R:R)
    double adjusted_sl, adjusted_tp;
    g_risk.AdjustSlTp(signal.entry_price, current_price,
                       signal.stop_price, signal.target_price,
                       adjusted_sl, adjusted_tp);

    adjusted_sl = NormalizeDouble(adjusted_sl, digits);
    adjusted_tp = NormalizeDouble(adjusted_tp, digits);

    // Prepare trade request
    MqlTradeRequest request = {};
    MqlTradeResult  result = {};

    request.action    = TRADE_ACTION_DEAL;
    request.symbol    = broker_symbol;
    request.volume    = lots;
    request.price     = NormalizeDouble(current_price, digits);
    request.sl        = adjusted_sl;
    request.tp        = adjusted_tp;
    request.deviation = (ulong)(g_settings.max_slippage_pips * 10);
    request.magic     = InpMagicNumber;
    request.comment   = "ATLAS|" + signal.id;
    request.type_filling = ORDER_FILLING_IOC;

    if(signal.direction == "buy")
        request.type = ORDER_TYPE_BUY;
    else
        request.type = ORDER_TYPE_SELL;

    // Send order
    if(!OrderSend(request, result))
    {
        int err = GetLastError();
        PrintFormat("[ATLAS EA] OrderSend failed for %s: error %d, retcode %u",
                    signal.id, err, result.retcode);
        g_http.SendFeedback(signal.id, "error", 0, 0, 0, 0, 0,
                            StringFormat("OrderSend error %d retcode %u",
                                         err, result.retcode));
        return false;
    }

    // Check result
    if(result.retcode != TRADE_RETCODE_DONE && result.retcode != TRADE_RETCODE_PLACED)
    {
        PrintFormat("[ATLAS EA] Order not filled for %s: retcode %u",
                    signal.id, result.retcode);
        g_http.SendFeedback(signal.id, "error", 0, 0, 0, 0, 0,
                            StringFormat("Order retcode %u", result.retcode));
        return false;
    }

    double fill_price = result.price;
    if(fill_price == 0) fill_price = current_price;

    PrintFormat("[ATLAS EA] Executed %s %s %.2f lots @ %.5f SL=%.5f TP=%.5f",
                signal.id, signal.direction, lots, fill_price,
                adjusted_sl, adjusted_tp);

    // Send accepted feedback
    g_http.SendFeedback(signal.id, "accepted", fill_price, lots);

    return true;
}

//+------------------------------------------------------------------+
//| Process signals from API response                                 |
//+------------------------------------------------------------------+
void ProcessSignals(AtlasSignal &signals[], int count)
{
    for(int i = 0; i < count; i++)
    {
        AtlasSignal sig = signals[i];

        // Skip if already processed
        if(IsSignalProcessed(sig.id))
            continue;

        // Skip if not active
        if(sig.status != "active")
        {
            MarkSignalProcessed(sig.id);
            continue;
        }

        // Check expiry
        if(g_risk.IsSignalExpired(sig.expires_at))
        {
            PrintFormat("[ATLAS EA] Signal %s expired", sig.id);
            MarkSignalProcessed(sig.id);
            continue;
        }

        // Check confidence threshold
        if(sig.confidence < g_settings.min_confidence)
        {
            PrintFormat("[ATLAS EA] Signal %s filtered: confidence %d < %d",
                        sig.id, sig.confidence, g_settings.min_confidence);
            g_http.SendFeedback(sig.id, "filtered", 0, 0, 0, 0, 0,
                                "", "low_confidence");
            MarkSignalProcessed(sig.id);
            continue;
        }

        // Check allowed symbols
        if(!g_risk.IsSymbolAllowed(sig.symbol, g_settings.allowed_symbols))
        {
            PrintFormat("[ATLAS EA] Signal %s filtered: symbol %s not allowed",
                        sig.id, sig.symbol);
            g_http.SendFeedback(sig.id, "filtered", 0, 0, 0, 0, 0,
                                "", "symbol_not_allowed");
            MarkSignalProcessed(sig.id);
            continue;
        }

        // Check max open trades
        int open_count = g_risk.CountOpenPositions(InpMagicNumber);
        if(open_count >= g_settings.max_open_trades)
        {
            PrintFormat("[ATLAS EA] Signal %s filtered: %d/%d positions open",
                        sig.id, open_count, g_settings.max_open_trades);
            g_http.SendFeedback(sig.id, "filtered", 0, 0, 0, 0, 0,
                                "", "max_trades_reached");
            MarkSignalProcessed(sig.id);
            continue;
        }

        // Check daily drawdown
        if(g_risk.IsDailyDrawdownBreached(g_settings.max_daily_dd_pct))
        {
            PrintFormat("[ATLAS EA] Signal %s filtered: daily DD %.1f%% >= %.1f%%",
                        sig.id, g_risk.GetDailyDrawdownPct(),
                        g_settings.max_daily_dd_pct);
            g_http.SendFeedback(sig.id, "filtered", 0, 0, 0, 0, 0,
                                "", "daily_drawdown_limit");
            MarkSignalProcessed(sig.id);
            continue;
        }

        // Check market hours
        string broker_symbol = BuildBrokerSymbol(sig.symbol);
        if(!g_risk.IsMarketOpen(broker_symbol))
        {
            // Don't mark as processed — try again next poll
            PrintFormat("[ATLAS EA] Market closed for %s, will retry", sig.symbol);
            continue;
        }

        // Execute or display popup
        if(g_settings.auto_execute)
        {
            if(ExecuteSignal(sig))
                MarkSignalProcessed(sig.id);
            else
                MarkSignalProcessed(sig.id); // Mark even on failure to avoid spam
        }
        else
        {
            // Manual mode: show alert and wait for user
            string msg = StringFormat(
                "ATLAS Signal: %s %s @ %.5f\nSL: %.5f  TP: %.5f\nConfidence: %d%%  R:R %.1f",
                sig.direction, sig.symbol, sig.entry_price,
                sig.stop_price, sig.target_price,
                sig.confidence, sig.rr_ratio);

            // Log and send notification
            PrintFormat("[ATLAS EA] Manual signal: %s", msg);
            Alert(msg);
            SendNotification(msg);

            // Send pending feedback
            g_http.SendFeedback(sig.id, "pending");
            MarkSignalProcessed(sig.id);
        }
    }
}

//+------------------------------------------------------------------+
//| Check for closed ATLAS positions and send feedback                |
//+------------------------------------------------------------------+
void CheckClosedPositions()
{
    // Check recent deals in history
    datetime from = TimeGMT() - 60; // Last 60 seconds
    datetime to = TimeGMT();

    if(!HistorySelect(from, to))
        return;

    int deals = HistoryDealsTotal();
    for(int i = deals - 1; i >= 0; i--)
    {
        ulong ticket = HistoryDealGetTicket(i);
        if(ticket == 0) continue;

        // Only our EA's deals
        if(HistoryDealGetInteger(ticket, DEAL_MAGIC) != InpMagicNumber)
            continue;

        // Only exit deals (DEAL_ENTRY_OUT)
        ENUM_DEAL_ENTRY entry = (ENUM_DEAL_ENTRY)HistoryDealGetInteger(ticket, DEAL_ENTRY);
        if(entry != DEAL_ENTRY_OUT)
            continue;

        // Extract signal ID from comment: "ATLAS|sig_xxxx"
        string comment = HistoryDealGetString(ticket, DEAL_COMMENT);
        string signal_id = "";
        int pipe_pos = StringFind(comment, "|");
        if(pipe_pos >= 0 && StringLen(comment) > (uint)(pipe_pos + 1))
            signal_id = StringSubstr(comment, pipe_pos + 1);

        if(signal_id == "" || StringFind(signal_id, SIGNAL_ID_PREFIX) != 0)
            continue;

        // Get deal details
        string symbol = HistoryDealGetString(ticket, DEAL_SYMBOL);
        double close_price = HistoryDealGetDouble(ticket, DEAL_PRICE);
        double profit = HistoryDealGetDouble(ticket, DEAL_PROFIT);
        double volume = HistoryDealGetDouble(ticket, DEAL_VOLUME);

        // We need the opening deal to calculate R
        // Find matching position ticket
        ulong pos_id = HistoryDealGetInteger(ticket, DEAL_POSITION_ID);

        // Look back further to find the entry deal
        HistorySelect(0, TimeGMT());
        int all_deals = HistoryDealsTotal();
        double open_price = 0;
        double stop_price = 0;
        string direction = "";

        for(int j = 0; j < all_deals; j++)
        {
            ulong d_ticket = HistoryDealGetTicket(j);
            if(d_ticket == 0) continue;
            if(HistoryDealGetInteger(d_ticket, DEAL_POSITION_ID) != (long)pos_id)
                continue;
            if((ENUM_DEAL_ENTRY)HistoryDealGetInteger(d_ticket, DEAL_ENTRY) == DEAL_ENTRY_IN)
            {
                open_price = HistoryDealGetDouble(d_ticket, DEAL_PRICE);
                ENUM_DEAL_TYPE dtype = (ENUM_DEAL_TYPE)HistoryDealGetInteger(d_ticket, DEAL_TYPE);
                direction = (dtype == DEAL_TYPE_BUY) ? "buy" : "sell";
                // SL from the comment or we estimate from signal data
                break;
            }
        }

        // Calculate P&L
        double pnl_pips = 0;
        double pnl_r = 0;
        if(open_price > 0)
        {
            pnl_pips = g_risk.CalculatePnlPips(symbol, direction,
                                                 open_price, close_price);
        }

        // Record realized P&L
        g_risk.RecordClosedPnl(profit);

        PrintFormat("[ATLAS EA] Position closed: %s %s %.2f lots, PnL: %.2f pips, $%.2f",
                    signal_id, direction, volume, pnl_pips, profit);

        // Send feedback with result
        g_http.SendFeedback(signal_id, "accepted", open_price, volume,
                            close_price, pnl_pips, pnl_r);
    }
}

//+------------------------------------------------------------------+
//| Expert initialization function                                    |
//+------------------------------------------------------------------+
int OnInit()
{
    PrintFormat("[ATLAS EA] Starting %s v%s", EA_NAME, EA_VERSION);

    //--- Validate API key format
    if(StringLen(InpApiKey) == 0 || StringFind(InpApiKey, "ak_") != 0)
    {
        PrintFormat("[ATLAS EA] Invalid API key format. Must start with 'ak_'");
        if(InpShowPanel)
        {
            g_panel.Init(InpPanelX, InpPanelY, InpPanelBg, InpPanelText);
            g_panel.Show();
            g_panel.UpdateConnection(false, "Invalid API key");
        }
        return INIT_FAILED;
    }

    //--- Initialize HTTP client
    g_http.Init(InpApiKey, InpServerUrl);

    //--- Initialize risk manager
    g_risk.Init();

    //--- Initialize settings with defaults from inputs
    g_settings.Reset();

    //--- Validate API key via server
    PrintFormat("[ATLAS EA] Validating API key...");
    if(!g_http.ValidateKey(g_user_plan, g_user_status, g_settings))
    {
        PrintFormat("[ATLAS EA] Authentication failed");
        if(InpShowPanel)
        {
            g_panel.Init(InpPanelX, InpPanelY, InpPanelBg, InpPanelText);
            g_panel.Show();
            g_panel.UpdateConnection(false, "Auth failed");
        }
        return INIT_FAILED;
    }

    g_authenticated = true;
    PrintFormat("[ATLAS EA] Authenticated. Plan: %s, Status: %s",
                g_user_plan, g_user_status);

    //--- Check if user account is active
    if(g_user_status != "active")
    {
        PrintFormat("[ATLAS EA] Account not active (status: %s)", g_user_status);
        if(InpShowPanel)
        {
            g_panel.Init(InpPanelX, InpPanelY, InpPanelBg, InpPanelText);
            g_panel.Show();
            g_panel.UpdateConnection(false, "Account " + g_user_status);
        }
        return INIT_FAILED;
    }

    //--- Initialize panel
    if(InpShowPanel)
    {
        g_panel.Init(InpPanelX, InpPanelY, InpPanelBg, InpPanelText);
        g_panel.Show();
        g_panel.UpdateConnection(true, "");
        g_panel.UpdatePlan(g_user_plan);
        g_panel.UpdateMode(g_settings.auto_execute);
    }

    //--- Initialize processed signals array
    ArrayResize(g_processed_ids, 0);
    g_processed_count = 0;

    //--- Start polling timer
    if(!EventSetTimer(InpPollInterval))
    {
        PrintFormat("[ATLAS EA] Failed to set timer");
        return INIT_FAILED;
    }

    PrintFormat("[ATLAS EA] Ready. Polling every %d seconds. Mode: %s",
                InpPollInterval,
                g_settings.auto_execute ? "AUTO" : "MANUAL");

    return INIT_SUCCEEDED;
}

//+------------------------------------------------------------------+
//| Expert deinitialization function                                  |
//+------------------------------------------------------------------+
void OnDeinit(const int reason)
{
    EventKillTimer();
    g_panel.Destroy();

    PrintFormat("[ATLAS EA] Stopped. Reason: %d. Processed %d signals.",
                reason, g_processed_count);
}

//+------------------------------------------------------------------+
//| Timer event — poll for signals                                    |
//+------------------------------------------------------------------+
void OnTimer()
{
    if(!g_authenticated)
        return;

    //--- Check daily reset
    g_risk.CheckDailyReset();

    //--- Fetch active signals
    AtlasSignal signals[];
    int count = 0;
    datetime server_time = 0;

    if(!g_http.FetchActiveSignals(signals, count, server_time))
    {
        g_poll_errors++;
        PrintFormat("[ATLAS EA] Poll failed (%d consecutive errors)", g_poll_errors);

        if(InpShowPanel)
            g_panel.UpdateConnection(false, StringFormat("Error (%d)", g_poll_errors));

        return;
    }

    // Reset error counter on success
    g_poll_errors = 0;
    g_last_poll = TimeGMT();
    g_signal_count = count;

    //--- Process signals
    if(count > 0)
        ProcessSignals(signals, count);

    //--- Update panel
    if(InpShowPanel)
    {
        g_panel.UpdateConnection(true, "");
        g_panel.UpdateLastPoll(g_last_poll);
        g_panel.UpdateSignals(g_signal_count);

        int open_positions = g_risk.CountOpenPositions(InpMagicNumber);
        g_panel.UpdatePositions(open_positions, g_settings.max_open_trades);

        double equity = AccountInfoDouble(ACCOUNT_EQUITY);
        double daily_pnl = equity - g_risk.DailyStartBalance();
        g_panel.UpdateDailyPnl(daily_pnl);

        double dd_pct = g_risk.GetDailyDrawdownPct();
        g_panel.UpdateDrawdown(dd_pct, g_settings.max_daily_dd_pct);
    }
}

//+------------------------------------------------------------------+
//| Trade event — detect position closes                              |
//+------------------------------------------------------------------+
void OnTrade()
{
    if(!g_authenticated)
        return;

    CheckClosedPositions();

    //--- Update panel position count
    if(InpShowPanel)
    {
        int open_positions = g_risk.CountOpenPositions(InpMagicNumber);
        g_panel.UpdatePositions(open_positions, g_settings.max_open_trades);
    }
}

//+------------------------------------------------------------------+
//| Chart event — handle user interactions (future: panel clicks)     |
//+------------------------------------------------------------------+
void OnChartEvent(const int id,
                  const long &lparam,
                  const double &dparam,
                  const string &sparam)
{
    // Reserved for future panel click handling
}
