//+------------------------------------------------------------------+
//|                                                  ATLAS_Http.mqh  |
//|                                    Copyright 2026, ATLAS FX Ltd. |
//|                                        https://www.atlasfxsignals.com   |
//+------------------------------------------------------------------+
#property copyright "Copyright 2026, ATLAS FX Ltd."
#property link      "https://www.atlasfxsignals.com"
#property strict

#ifndef ATLAS_HTTP_MQH
#define ATLAS_HTTP_MQH

#include "ATLAS_Config.mqh"
#include "ATLAS_Json.mqh"

//+------------------------------------------------------------------+
//| HTTP client for ATLAS API communication                           |
//+------------------------------------------------------------------+
class AtlasHttp
{
private:
    string m_api_key;
    string m_base_url;
    int    m_timeout;
    int    m_last_status;
    string m_last_error;

    //--- Build common headers string
    string BuildHeaders(string content_type = "")
    {
        string hdrs = "";
        hdrs += "X-API-Key: " + m_api_key + "\r\n";
        hdrs += "User-Agent: " + EA_USER_AGENT + "\r\n";
        hdrs += "Accept: application/json\r\n";
        if(content_type != "")
            hdrs += "Content-Type: " + content_type + "\r\n";
        return hdrs;
    }

    //--- Execute WebRequest with retry logic
    HttpResponse DoRequest(string method, string url, string body = "")
    {
        HttpResponse response;
        response.Reset();

        char  data[];
        char  result[];
        string result_headers;

        // Prepare body data (use StringLen to exclude null terminator)
        if(body != "")
            StringToCharArray(body, data, 0, StringLen(body), CP_UTF8);
        else
            ArrayResize(data, 0);

        // Determine content type
        string content_type = "";
        if(method == "POST" || method == "PUT" || method == "PATCH")
            content_type = "application/json";

        string headers = BuildHeaders(content_type);

        // Attempt request with retry
        for(int attempt = 0; attempt <= HTTP_MAX_RETRIES; attempt++)
        {
            if(attempt > 0)
            {
                PrintFormat("[ATLAS HTTP] Retry %d for %s %s", attempt, method, url);
                Sleep(HTTP_RETRY_DELAY_MS);
            }

            ResetLastError();
            int status = WebRequest(method, url, headers, m_timeout, data,
                                    result, result_headers);

            if(status == -1)
            {
                int err = GetLastError();
                response.status_code = -1;
                response.error_message = StringFormat("WebRequest failed (error %d). "
                    "Ensure %s is whitelisted in Tools > Options > Expert Advisors.",
                    err, InpServerUrl);
                PrintFormat("[ATLAS HTTP] WebRequest error %d on %s %s", err, method, url);

                if(attempt < HTTP_MAX_RETRIES)
                    continue;
                return response;
            }

            // Convert response body from char[] to string
            response.status_code = status;
            response.headers = result_headers;

            if(ArraySize(result) > 0)
                response.body = CharArrayToString(result, 0, WHOLE_ARRAY, CP_UTF8);
            else
                response.body = "";

            // Success (2xx)
            if(status >= 200 && status < 300)
            {
                response.success = true;
                m_last_status = status;
                m_last_error = "";
                return response;
            }

            // Rate limited (429) — retry
            if(status == 429 && attempt < HTTP_MAX_RETRIES)
            {
                PrintFormat("[ATLAS HTTP] Rate limited (429) on %s %s, retrying...",
                            method, url);
                Sleep(2000); // Wait 2s before retry on rate limit
                continue;
            }

            // Server error (5xx) — retry
            if(status >= 500 && attempt < HTTP_MAX_RETRIES)
            {
                PrintFormat("[ATLAS HTTP] Server error (%d) on %s %s, retrying...",
                            status, method, url);
                continue;
            }

            // Non-retryable error (4xx other than 429, or exhausted retries)
            response.success = false;
            response.error_message = StringFormat("HTTP %d on %s %s", status, method, url);

            // Try to parse error from response body
            if(response.body != "")
            {
                JsonParser parser;
                JsonValue *root = parser.Parse(response.body);
                if(root != NULL && root.IsObject())
                {
                    JsonValue *err_obj = root.Get("error");
                    if(err_obj != NULL && err_obj.IsObject())
                    {
                        response.error_message = err_obj.GetString("message",
                                                    response.error_message);
                    }
                }
                if(root != NULL) delete root;
            }

            m_last_status = status;
            m_last_error = response.error_message;
            PrintFormat("[ATLAS HTTP] %s", response.error_message);
            return response;
        }

        return response;
    }

public:
    AtlasHttp()
        : m_api_key(""), m_base_url(""), m_timeout(HTTP_TIMEOUT_MS),
          m_last_status(0), m_last_error("")
    {}

    //--- Initialize with API key and base URL
    void Init(string api_key, string server_url)
    {
        m_api_key = api_key;
        m_base_url = server_url;

        // Strip trailing slash
        if(StringGetCharacter(m_base_url, StringLen(m_base_url) - 1) == '/')
            m_base_url = StringSubstr(m_base_url, 0, StringLen(m_base_url) - 1);

        // Ensure /v1 suffix
        if(StringFind(m_base_url, "/v1") < 0)
            m_base_url += "/v1";
    }

    //--- GET request
    HttpResponse Get(string endpoint)
    {
        string url = m_base_url + endpoint;
        return DoRequest("GET", url);
    }

    //--- POST request with JSON body
    HttpResponse Post(string endpoint, string json_body = "{}")
    {
        string url = m_base_url + endpoint;
        return DoRequest("POST", url, json_body);
    }

    //--- Validate API key via POST /auth/validate
    bool ValidateKey(string &out_plan, string &out_status, AtlasSettings &settings)
    {
        HttpResponse resp = Post("/auth/validate", "{}");
        if(!resp.success)
        {
            PrintFormat("[ATLAS HTTP] Auth validation failed: %s", resp.error_message);
            return false;
        }

        JsonParser parser;
        JsonValue *root = parser.Parse(resp.body);
        if(root == NULL || root.IsError())
        {
            PrintFormat("[ATLAS HTTP] Failed to parse auth response");
            if(root != NULL) delete root;
            return false;
        }

        bool valid = root.GetBool("valid", false);
        if(!valid)
        {
            string err = root.GetString("error", "Unknown auth error");
            PrintFormat("[ATLAS HTTP] Auth invalid: %s", err);
            delete root;
            return false;
        }

        out_plan = root.GetString("plan", "free");
        out_status = root.GetString("status", "active");

        // Parse settings if present
        JsonValue *s = root.Get("settings");
        if(s != NULL && s.IsObject())
        {
            settings.min_confidence         = (int)s.GetInt("min_confidence", InpMinConfidence);
            settings.max_risk_pct           = s.GetDouble("max_risk_pct", InpMaxRiskPercent);
            settings.max_daily_dd_pct       = s.GetDouble("max_daily_dd_pct", InpMaxDailyDD);
            settings.max_weekly_dd_pct      = s.GetDouble("max_weekly_dd_pct", 5.0);
            settings.max_open_trades        = (int)s.GetInt("max_open_trades", InpMaxOpenTrades);
            settings.max_slippage_pips      = s.GetDouble("max_slippage_pips", InpMaxSlippagePips);
            settings.max_entry_deviation_pct = s.GetDouble("max_entry_deviation_pct", 0.5);
            settings.auto_execute           = s.GetBool("auto_execute", InpAutoExecute);

            // Parse allowed symbols array
            JsonValue *syms = s.Get("allowed_symbols");
            if(syms != NULL && syms.IsArray())
            {
                int count = syms.Size();
                ArrayResize(settings.allowed_symbols, count);
                for(int i = 0; i < count; i++)
                {
                    JsonValue *sym = syms.At(i);
                    if(sym != NULL && sym.IsString())
                        settings.allowed_symbols[i] = sym.ToStr();
                }
            }
        }

        delete root;
        return true;
    }

    //--- Fetch active signals
    bool FetchActiveSignals(AtlasSignal &signals[], int &count, datetime &server_time)
    {
        count = 0;
        HttpResponse resp = Get("/signals/active");
        if(!resp.success)
        {
            PrintFormat("[ATLAS HTTP] Failed to fetch signals: %s", resp.error_message);
            return false;
        }

        JsonParser parser;
        JsonValue *root = parser.Parse(resp.body);
        if(root == NULL || root.IsError())
        {
            PrintFormat("[ATLAS HTTP] Failed to parse signals response");
            if(root != NULL) delete root;
            return false;
        }

        // Parse server_time
        string st = root.GetString("server_time", "");
        if(st != "")
            server_time = ParseIso8601(st);

        // Parse signals array
        JsonValue *sigs = root.Get("signals");
        if(sigs == NULL || !sigs.IsArray())
        {
            delete root;
            return true; // Valid response with no signals
        }

        count = sigs.Size();
        ArrayResize(signals, count);

        for(int i = 0; i < count; i++)
        {
            JsonValue *sig = sigs.At(i);
            if(sig == NULL || !sig.IsObject())
                continue;

            signals[i].id           = sig.GetString("id", "");
            signals[i].symbol       = sig.GetString("symbol", "");
            signals[i].direction    = sig.GetString("direction", "");
            signals[i].confidence   = (int)sig.GetInt("confidence", 0);
            signals[i].regime       = sig.GetString("regime", "");
            signals[i].entry_price  = sig.GetDouble("entry_price", 0);
            signals[i].stop_price   = sig.GetDouble("stop_price", 0);
            signals[i].target_price = sig.GetDouble("target_price", 0);
            signals[i].risk_pips    = sig.GetDouble("risk_pips", 0);
            signals[i].reward_pips  = sig.GetDouble("reward_pips", 0);
            signals[i].rr_ratio     = sig.GetDouble("rr_ratio", 0);
            signals[i].status       = sig.GetString("status", "");
            signals[i].expires_at   = ParseIso8601(sig.GetString("expires_at", ""));
            signals[i].created_at   = ParseIso8601(sig.GetString("created_at", ""));
        }

        delete root;
        return true;
    }

    //--- Send signal feedback
    bool SendFeedback(string signal_id, string action, double executed_price = 0,
                      double executed_size = 0, double close_price = 0,
                      double pnl_pips = 0, double pnl_r = 0,
                      string error_msg = "", string filter_reason = "")
    {
        JsonBuilder jb;
        jb.AddString("signal_id", signal_id);
        jb.AddString("action", action);
        jb.AddString("execution_type", "self_managed");

        if(executed_price > 0)  jb.AddDouble("executed_price", executed_price);
        if(executed_size > 0)   jb.AddDouble("executed_size", executed_size);
        if(close_price > 0)     jb.AddDouble("close_price", close_price);
        if(pnl_pips != 0)       jb.AddDouble("pnl_pips", pnl_pips, 2);
        if(pnl_r != 0)          jb.AddDouble("pnl_r", pnl_r, 2);
        if(error_msg != "")     jb.AddString("error_message", error_msg);
        if(filter_reason != "") jb.AddString("filter_reason", filter_reason);

        HttpResponse resp = Post("/signals/feedback", jb.Build());
        if(!resp.success)
        {
            PrintFormat("[ATLAS HTTP] Feedback failed for %s: %s",
                        signal_id, resp.error_message);
            return false;
        }

        return true;
    }

    //--- Getters
    int    LastStatus()       const { return m_last_status; }
    string LastError()        const { return m_last_error; }
};

#endif // ATLAS_HTTP_MQH
