//+------------------------------------------------------------------+
//|                                                  ATLAS_Json.mqh  |
//|                                    Copyright 2026, ATLAS FX Ltd. |
//|                                        https://www.atlasfxsignals.com   |
//+------------------------------------------------------------------+
#property copyright "Copyright 2026, ATLAS FX Ltd."
#property link      "https://www.atlasfxsignals.com"
#property strict

#ifndef ATLAS_JSON_MQH
#define ATLAS_JSON_MQH

//+------------------------------------------------------------------+
//| JSON value types                                                  |
//+------------------------------------------------------------------+
enum JSON_TYPE
{
    JSON_NULL,
    JSON_BOOL,
    JSON_INT,
    JSON_DOUBLE,
    JSON_STRING,
    JSON_ARRAY,
    JSON_OBJECT,
    JSON_ERROR
};

//+------------------------------------------------------------------+
//| Forward declaration                                               |
//+------------------------------------------------------------------+
class JsonValue;

//+------------------------------------------------------------------+
//| Key-value pair for JSON objects                                   |
//+------------------------------------------------------------------+
struct JsonPair
{
    string     key;
    JsonValue *value;
};

//+------------------------------------------------------------------+
//| JSON Value — represents any JSON type                             |
//+------------------------------------------------------------------+
class JsonValue
{
private:
    JSON_TYPE   m_type;
    string      m_string_val;
    double      m_double_val;
    long        m_int_val;
    bool        m_bool_val;

    // Object: array of key-value pairs
    JsonPair    m_pairs[];
    int         m_pair_count;

    // Array: array of values
    JsonValue  *m_items[];
    int         m_item_count;

public:
    JsonValue()
        : m_type(JSON_NULL), m_string_val(""), m_double_val(0),
          m_int_val(0), m_bool_val(false), m_pair_count(0), m_item_count(0)
    {}

    ~JsonValue()
    {
        Clear();
    }

    void Clear()
    {
        for(int i = 0; i < m_pair_count; i++)
        {
            if(m_pairs[i].value != NULL)
            {
                delete m_pairs[i].value;
                m_pairs[i].value = NULL;
            }
        }
        m_pair_count = 0;
        ArrayResize(m_pairs, 0);

        for(int i = 0; i < m_item_count; i++)
        {
            if(m_items[i] != NULL)
            {
                delete m_items[i];
                m_items[i] = NULL;
            }
        }
        m_item_count = 0;
        ArrayResize(m_items, 0);

        m_type = JSON_NULL;
    }

    //--- Type getters
    JSON_TYPE Type()        const { return m_type; }
    bool IsNull()           const { return m_type == JSON_NULL; }
    bool IsBool()           const { return m_type == JSON_BOOL; }
    bool IsInt()            const { return m_type == JSON_INT; }
    bool IsDouble()         const { return m_type == JSON_DOUBLE; }
    bool IsNumber()         const { return m_type == JSON_INT || m_type == JSON_DOUBLE; }
    bool IsString()         const { return m_type == JSON_STRING; }
    bool IsArray()          const { return m_type == JSON_ARRAY; }
    bool IsObject()         const { return m_type == JSON_OBJECT; }
    bool IsError()          const { return m_type == JSON_ERROR; }

    //--- Value getters
    string ToStr()          const { return m_string_val; }
    double ToDouble()       const
    {
        if(m_type == JSON_INT) return (double)m_int_val;
        return m_double_val;
    }
    long   ToInt()          const
    {
        if(m_type == JSON_DOUBLE) return (long)m_double_val;
        return m_int_val;
    }
    bool   ToBool()         const { return m_bool_val; }

    //--- Object access
    int    Size() const
    {
        if(m_type == JSON_ARRAY) return m_item_count;
        if(m_type == JSON_OBJECT) return m_pair_count;
        return 0;
    }

    //--- Get object value by key (returns NULL if not found)
    JsonValue *Get(string key) const
    {
        if(m_type != JSON_OBJECT)
            return NULL;
        for(int i = 0; i < m_pair_count; i++)
        {
            if(m_pairs[i].key == key)
                return m_pairs[i].value;
        }
        return NULL;
    }

    //--- Check if object has a key
    bool Has(string key) const
    {
        return Get(key) != NULL;
    }

    //--- Get string from object key with default
    string GetString(string key, string default_val = "") const
    {
        JsonValue *v = Get(key);
        if(v == NULL || v.IsNull()) return default_val;
        return v.ToStr();
    }

    //--- Get int from object key with default
    long GetInt(string key, long default_val = 0) const
    {
        JsonValue *v = Get(key);
        if(v == NULL || v.IsNull()) return default_val;
        return v.ToInt();
    }

    //--- Get double from object key with default
    double GetDouble(string key, double default_val = 0.0) const
    {
        JsonValue *v = Get(key);
        if(v == NULL || v.IsNull()) return default_val;
        return v.ToDouble();
    }

    //--- Get bool from object key with default
    bool GetBool(string key, bool default_val = false) const
    {
        JsonValue *v = Get(key);
        if(v == NULL || v.IsNull()) return default_val;
        return v.ToBool();
    }

    //--- Array access by index
    JsonValue *At(int index) const
    {
        if(m_type != JSON_ARRAY || index < 0 || index >= m_item_count)
            return NULL;
        return m_items[index];
    }

    //--- Get key at index (for object iteration)
    string KeyAt(int index) const
    {
        if(m_type != JSON_OBJECT || index < 0 || index >= m_pair_count)
            return "";
        return m_pairs[index].key;
    }

    //--- Get value at index (for object iteration)
    JsonValue *ValueAt(int index) const
    {
        if(m_type != JSON_OBJECT || index < 0 || index >= m_pair_count)
            return NULL;
        return m_pairs[index].value;
    }

    //--- Setters (used by parser)
    void SetNull()           { m_type = JSON_NULL; }
    void SetError(string msg){ m_type = JSON_ERROR; m_string_val = msg; }
    void SetBool(bool v)     { m_type = JSON_BOOL; m_bool_val = v; }
    void SetInt(long v)      { m_type = JSON_INT; m_int_val = v; }
    void SetDouble(double v) { m_type = JSON_DOUBLE; m_double_val = v; }
    void SetString(string v) { m_type = JSON_STRING; m_string_val = v; }

    void SetArray()
    {
        m_type = JSON_ARRAY;
        m_item_count = 0;
    }

    void SetObject()
    {
        m_type = JSON_OBJECT;
        m_pair_count = 0;
    }

    void AddItem(JsonValue *item)
    {
        int new_size = m_item_count + 1;
        ArrayResize(m_items, new_size);
        m_items[m_item_count] = item;
        m_item_count = new_size;
    }

    void AddPair(string key, JsonValue *value)
    {
        int new_size = m_pair_count + 1;
        ArrayResize(m_pairs, new_size);
        m_pairs[m_pair_count].key = key;
        m_pairs[m_pair_count].value = value;
        m_pair_count = new_size;
    }
};

//+------------------------------------------------------------------+
//| JSON Parser                                                       |
//+------------------------------------------------------------------+
class JsonParser
{
private:
    string m_json;
    int    m_pos;
    int    m_len;

    //--- Skip whitespace
    void SkipWhitespace()
    {
        while(m_pos < m_len)
        {
            ushort ch = StringGetCharacter(m_json, m_pos);
            if(ch == ' ' || ch == '\t' || ch == '\r' || ch == '\n')
                m_pos++;
            else
                break;
        }
    }

    //--- Peek current character
    ushort Peek()
    {
        if(m_pos >= m_len) return 0;
        return StringGetCharacter(m_json, m_pos);
    }

    //--- Consume current character and advance
    ushort Next()
    {
        if(m_pos >= m_len) return 0;
        ushort ch = StringGetCharacter(m_json, m_pos);
        m_pos++;
        return ch;
    }

    //--- Parse a string value (after opening quote is consumed)
    string ParseString()
    {
        string result = "";
        while(m_pos < m_len)
        {
            ushort ch = Next();
            if(ch == '"')
                return result;
            if(ch == '\\')
            {
                ushort esc = Next();
                switch(esc)
                {
                    case '"':  result += "\""; break;
                    case '\\': result += "\\"; break;
                    case '/':  result += "/";  break;
                    case 'b':  result += "\b"; break;
                    case 'f':  result += "\f"; break;
                    case 'n':  result += "\n"; break;
                    case 'r':  result += "\r"; break;
                    case 't':  result += "\t"; break;
                    case 'u':
                    {
                        // Unicode escape: \uXXXX — read 4 hex digits
                        string hex = "";
                        for(int i = 0; i < 4 && m_pos < m_len; i++)
                            hex += ShortToString(Next());
                        long code = 0;
                        for(int i = 0; i < StringLen(hex); i++)
                        {
                            ushort h = StringGetCharacter(hex, i);
                            code *= 16;
                            if(h >= '0' && h <= '9')      code += h - '0';
                            else if(h >= 'a' && h <= 'f') code += h - 'a' + 10;
                            else if(h >= 'A' && h <= 'F') code += h - 'A' + 10;
                        }
                        result += ShortToString((ushort)code);
                        break;
                    }
                    default:
                        result += ShortToString(esc);
                        break;
                }
            }
            else
            {
                result += ShortToString(ch);
            }
        }
        return result; // Unterminated string
    }

    //--- Parse a number
    JsonValue *ParseNumber()
    {
        int start = m_pos;
        bool is_float = false;

        // Handle negative
        if(Peek() == '-') m_pos++;

        // Integer part
        while(m_pos < m_len)
        {
            ushort ch = Peek();
            if(ch >= '0' && ch <= '9')
                m_pos++;
            else
                break;
        }

        // Decimal part
        if(m_pos < m_len && Peek() == '.')
        {
            is_float = true;
            m_pos++;
            while(m_pos < m_len)
            {
                ushort ch = Peek();
                if(ch >= '0' && ch <= '9')
                    m_pos++;
                else
                    break;
            }
        }

        // Exponent part
        if(m_pos < m_len && (Peek() == 'e' || Peek() == 'E'))
        {
            is_float = true;
            m_pos++;
            if(m_pos < m_len && (Peek() == '+' || Peek() == '-'))
                m_pos++;
            while(m_pos < m_len)
            {
                ushort ch = Peek();
                if(ch >= '0' && ch <= '9')
                    m_pos++;
                else
                    break;
            }
        }

        string num_str = StringSubstr(m_json, start, m_pos - start);
        JsonValue *val = new JsonValue();

        if(is_float)
            val.SetDouble(StringToDouble(num_str));
        else
            val.SetInt((long)StringToInteger(num_str));

        return val;
    }

    //--- Parse any value
    JsonValue *ParseValue()
    {
        SkipWhitespace();

        if(m_pos >= m_len)
        {
            JsonValue *val = new JsonValue();
            val.SetError("Unexpected end of input");
            return val;
        }

        ushort ch = Peek();

        // String
        if(ch == '"')
        {
            m_pos++; // consume opening quote
            JsonValue *val = new JsonValue();
            val.SetString(ParseString());
            return val;
        }

        // Object
        if(ch == '{')
        {
            m_pos++; // consume {
            JsonValue *obj = new JsonValue();
            obj.SetObject();
            SkipWhitespace();
            if(Peek() == '}')
            {
                m_pos++;
                return obj;
            }

            while(m_pos < m_len)
            {
                SkipWhitespace();
                // Parse key
                if(Peek() != '"')
                {
                    obj.SetError("Expected string key in object");
                    return obj;
                }
                m_pos++; // consume "
                string key = ParseString();

                SkipWhitespace();
                if(Next() != ':')
                {
                    obj.SetError("Expected ':' after key");
                    return obj;
                }

                // Parse value
                JsonValue *val = ParseValue();
                obj.AddPair(key, val);

                SkipWhitespace();
                ushort sep = Peek();
                if(sep == ',')
                {
                    m_pos++;
                    continue;
                }
                else if(sep == '}')
                {
                    m_pos++;
                    break;
                }
                else
                {
                    obj.SetError("Expected ',' or '}' in object");
                    return obj;
                }
            }
            return obj;
        }

        // Array
        if(ch == '[')
        {
            m_pos++; // consume [
            JsonValue *arr = new JsonValue();
            arr.SetArray();
            SkipWhitespace();
            if(Peek() == ']')
            {
                m_pos++;
                return arr;
            }

            while(m_pos < m_len)
            {
                JsonValue *item = ParseValue();
                arr.AddItem(item);

                SkipWhitespace();
                ushort sep = Peek();
                if(sep == ',')
                {
                    m_pos++;
                    continue;
                }
                else if(sep == ']')
                {
                    m_pos++;
                    break;
                }
                else
                {
                    arr.SetError("Expected ',' or ']' in array");
                    return arr;
                }
            }
            return arr;
        }

        // Number
        if(ch == '-' || (ch >= '0' && ch <= '9'))
        {
            return ParseNumber();
        }

        // true
        if(ch == 't' && m_pos + 3 < m_len &&
           StringSubstr(m_json, m_pos, 4) == "true")
        {
            m_pos += 4;
            JsonValue *val = new JsonValue();
            val.SetBool(true);
            return val;
        }

        // false
        if(ch == 'f' && m_pos + 4 < m_len &&
           StringSubstr(m_json, m_pos, 5) == "false")
        {
            m_pos += 5;
            JsonValue *val = new JsonValue();
            val.SetBool(false);
            return val;
        }

        // null
        if(ch == 'n' && m_pos + 3 < m_len &&
           StringSubstr(m_json, m_pos, 4) == "null")
        {
            m_pos += 4;
            JsonValue *val = new JsonValue();
            val.SetNull();
            return val;
        }

        // Unknown
        JsonValue *val = new JsonValue();
        val.SetError("Unexpected character: " + ShortToString(ch));
        return val;
    }

public:
    //--- Parse a JSON string and return root value
    //--- CALLER is responsible for deleting the returned pointer
    JsonValue *Parse(string json_str)
    {
        m_json = json_str;
        m_pos = 0;
        m_len = StringLen(json_str);

        if(m_len == 0)
        {
            JsonValue *val = new JsonValue();
            val.SetError("Empty input");
            return val;
        }

        JsonValue *root = ParseValue();
        return root;
    }
};

//+------------------------------------------------------------------+
//| Convenience: Build a JSON string for POST bodies                  |
//+------------------------------------------------------------------+
class JsonBuilder
{
private:
    string m_buffer;
    bool   m_first;

public:
    JsonBuilder() : m_buffer("{"), m_first(true) {}

    void AddString(string key, string value)
    {
        if(!m_first) m_buffer += ",";
        m_first = false;
        m_buffer += "\"" + key + "\":\"" + EscapeString(value) + "\"";
    }

    void AddInt(string key, long value)
    {
        if(!m_first) m_buffer += ",";
        m_first = false;
        m_buffer += "\"" + key + "\":" + IntegerToString(value);
    }

    void AddDouble(string key, double value, int digits = 5)
    {
        if(!m_first) m_buffer += ",";
        m_first = false;
        m_buffer += "\"" + key + "\":" + DoubleToString(value, digits);
    }

    void AddBool(string key, bool value)
    {
        if(!m_first) m_buffer += ",";
        m_first = false;
        m_buffer += "\"" + key + "\":" + (value ? "true" : "false");
    }

    void AddNull(string key)
    {
        if(!m_first) m_buffer += ",";
        m_first = false;
        m_buffer += "\"" + key + "\":null";
    }

    string Build()
    {
        return m_buffer + "}";
    }

    static string EscapeString(string s)
    {
        string result = "";
        int len = StringLen(s);
        for(int i = 0; i < len; i++)
        {
            ushort ch = StringGetCharacter(s, i);
            switch(ch)
            {
                case '"':  result += "\\\""; break;
                case '\\': result += "\\\\"; break;
                case '\n': result += "\\n";  break;
                case '\r': result += "\\r";  break;
                case '\t': result += "\\t";  break;
                default:
                    if(ch < 0x20)
                        result += StringFormat("\\u%04x", ch);
                    else
                        result += ShortToString(ch);
                    break;
            }
        }
        return result;
    }
};

#endif // ATLAS_JSON_MQH
