//+------------------------------------------------------------------+
//|                                                  ATLAS_Risk.mqh  |
//|                                    Copyright 2026, ATLAS FX Ltd. |
//|                                        https://www.atlasfxsignals.com   |
//+------------------------------------------------------------------+
#property copyright "Copyright 2026, ATLAS FX Ltd."
#property link      "https://www.atlasfxsignals.com"
#property strict

#ifndef ATLAS_RISK_MQH
#define ATLAS_RISK_MQH

#include "ATLAS_Config.mqh"

//+------------------------------------------------------------------+
//| Risk management and position sizing                               |
//+------------------------------------------------------------------+
class AtlasRisk
{
private:
    double m_daily_start_balance;     // Balance at start of day (UTC)
    double m_daily_realized_pnl;      // Realized P&L today
    int    m_daily_reset_day;         // Day number for daily reset
    int    m_open_atlas_positions;    // Current ATLAS position count

    //--- Get tick value in account currency for 1 lot
    double GetTickValuePerLot(string symbol)
    {
        double tick_value = SymbolInfoDouble(symbol, SYMBOL_TRADE_TICK_VALUE);
        double tick_size = SymbolInfoDouble(symbol, SYMBOL_TRADE_TICK_SIZE);

        if(tick_size == 0 || tick_value == 0)
        {
            PrintFormat("[ATLAS RISK] Cannot get tick info for %s", symbol);
            return 0;
        }

        return tick_value;
    }

public:
    AtlasRisk()
        : m_daily_start_balance(0), m_daily_realized_pnl(0),
          m_daily_reset_day(0), m_open_atlas_positions(0)
    {}

    //--- Initialize daily tracking
    void Init()
    {
        m_daily_start_balance = AccountInfoDouble(ACCOUNT_BALANCE);
        m_daily_realized_pnl = 0;
        MqlDateTime now;
        TimeToStruct(TimeGMT(), now);
        m_daily_reset_day = now.day_of_year;
    }

    //--- Check and reset daily counters if new day (UTC)
    void CheckDailyReset()
    {
        MqlDateTime now;
        TimeToStruct(TimeGMT(), now);
        if(now.day_of_year != m_daily_reset_day)
        {
            m_daily_start_balance = AccountInfoDouble(ACCOUNT_BALANCE);
            m_daily_realized_pnl = 0;
            m_daily_reset_day = now.day_of_year;
            PrintFormat("[ATLAS RISK] Daily reset. Start balance: %.2f",
                        m_daily_start_balance);
        }
    }

    //--- Record realized P&L from a closed position
    void RecordClosedPnl(double pnl)
    {
        m_daily_realized_pnl += pnl;
    }

    //+------------------------------------------------------------------+
    //| Calculate position size in lots                                   |
    //|                                                                   |
    //| Formula: lots = (balance * risk_pct / 100) /                      |
    //|                 (stop_distance_points * tick_value_per_lot /       |
    //|                  tick_size)                                        |
    //|                                                                   |
    //| Where stop_distance_points = |entry_price - stop_price|           |
    //+------------------------------------------------------------------+
    double CalculateLotSize(string symbol, double entry_price, double stop_price,
                            double risk_pct)
    {
        if(entry_price <= 0 || stop_price <= 0 || risk_pct <= 0)
            return 0;

        double balance = AccountInfoDouble(ACCOUNT_BALANCE);
        if(balance <= 0) return 0;

        // Risk amount in account currency
        double risk_amount = balance * risk_pct / 100.0;

        // Stop distance in price points
        double stop_distance = MathAbs(entry_price - stop_price);
        if(stop_distance == 0) return 0;

        // Get tick value and tick size
        double tick_value = SymbolInfoDouble(symbol, SYMBOL_TRADE_TICK_VALUE);
        double tick_size = SymbolInfoDouble(symbol, SYMBOL_TRADE_TICK_SIZE);

        if(tick_value <= 0 || tick_size <= 0)
        {
            PrintFormat("[ATLAS RISK] Invalid tick info for %s: tv=%.5f ts=%.5f",
                        symbol, tick_value, tick_size);
            return 0;
        }

        // Cost per lot for our stop distance
        double cost_per_lot = (stop_distance / tick_size) * tick_value;
        if(cost_per_lot <= 0) return 0;

        double lots = risk_amount / cost_per_lot;

        // Normalize to broker constraints
        lots = NormalizeLots(symbol, lots);

        return lots;
    }

    //+------------------------------------------------------------------+
    //| Normalize lot size to broker min/max/step                         |
    //+------------------------------------------------------------------+
    double NormalizeLots(string symbol, double lots)
    {
        double min_lot = SymbolInfoDouble(symbol, SYMBOL_VOLUME_MIN);
        double max_lot = SymbolInfoDouble(symbol, SYMBOL_VOLUME_MAX);
        double step = SymbolInfoDouble(symbol, SYMBOL_VOLUME_STEP);

        if(step <= 0) step = 0.01;
        if(min_lot <= 0) min_lot = 0.01;
        if(max_lot <= 0) max_lot = 100.0;

        // Round down to step
        lots = MathFloor(lots / step) * step;

        // Clamp to range
        if(lots < min_lot) lots = 0; // Cannot meet minimum — signal skip
        if(lots > max_lot) lots = max_lot;

        // Round to avoid floating point issues
        int digits = (int)MathCeil(-MathLog10(step));
        lots = NormalizeDouble(lots, digits);

        return lots;
    }

    //+------------------------------------------------------------------+
    //| Adjust SL and TP to preserve R:R when actual fill differs from   |
    //| signal entry price                                                |
    //|                                                                   |
    //| If signal says entry=2341.50, SL=2326.20, TP=2372.10 (BUY)      |
    //| But we fill at 2343.20 (1.70 higher), then:                      |
    //|   SL = 2326.20 + 1.70 = 2327.90                                 |
    //|   TP = 2372.10 + 1.70 = 2373.80                                 |
    //| This preserves exact same risk/reward in points                   |
    //+------------------------------------------------------------------+
    void AdjustSlTp(double signal_entry, double actual_entry,
                    double signal_sl, double signal_tp,
                    double &adjusted_sl, double &adjusted_tp)
    {
        double slippage = actual_entry - signal_entry;
        adjusted_sl = signal_sl + slippage;
        adjusted_tp = signal_tp + slippage;
    }

    //+------------------------------------------------------------------+
    //| Check if entry price deviation is within acceptable slippage      |
    //+------------------------------------------------------------------+
    bool IsSlippageAcceptable(string symbol, double signal_entry, double current_price,
                              double max_slippage_pips)
    {
        double point = SymbolInfoDouble(symbol, SYMBOL_POINT);
        int digits = (int)SymbolInfoInteger(symbol, SYMBOL_DIGITS);

        if(point <= 0) return false;

        // Convert slippage pips to points
        // For 5-digit brokers (EURUSD 1.12345), 1 pip = 10 points
        // For 3-digit brokers (XAUUSD 2341.50), 1 pip = 10 points
        // For 2-digit brokers (USDJPY 150.12), 1 pip = 10 points (if 3-digit)
        double pip_size = point;
        if(digits == 5 || digits == 3)
            pip_size = point * 10;

        double deviation_pips = MathAbs(current_price - signal_entry) / pip_size;
        return deviation_pips <= max_slippage_pips;
    }

    //+------------------------------------------------------------------+
    //| Calculate current daily drawdown percentage                       |
    //+------------------------------------------------------------------+
    double GetDailyDrawdownPct()
    {
        CheckDailyReset();

        if(m_daily_start_balance <= 0) return 0;

        // Include unrealized P&L from open positions
        double equity = AccountInfoDouble(ACCOUNT_EQUITY);
        double current_pnl = equity - m_daily_start_balance;

        if(current_pnl >= 0) return 0; // No drawdown

        return MathAbs(current_pnl) / m_daily_start_balance * 100.0;
    }

    //+------------------------------------------------------------------+
    //| Check if daily drawdown limit is breached                         |
    //+------------------------------------------------------------------+
    bool IsDailyDrawdownBreached(double max_dd_pct)
    {
        return GetDailyDrawdownPct() >= max_dd_pct;
    }

    //+------------------------------------------------------------------+
    //| Count open positions by magic number                              |
    //+------------------------------------------------------------------+
    int CountOpenPositions(int magic_number)
    {
        int count = 0;
        int total = PositionsTotal();
        for(int i = 0; i < total; i++)
        {
            ulong ticket = PositionGetTicket(i);
            if(ticket == 0) continue;

            if(PositionGetInteger(POSITION_MAGIC) == magic_number)
                count++;
        }
        m_open_atlas_positions = count;
        return count;
    }

    //+------------------------------------------------------------------+
    //| Check if symbol is in allowed list                                |
    //+------------------------------------------------------------------+
    bool IsSymbolAllowed(string signal_symbol, const string &allowed[])
    {
        int count = ArraySize(allowed);
        if(count == 0) return true; // Empty list = all allowed

        for(int i = 0; i < count; i++)
        {
            if(allowed[i] == signal_symbol)
                return true;
        }
        return false;
    }

    //+------------------------------------------------------------------+
    //| Check if signal has expired (compare against UTC time)            |
    //+------------------------------------------------------------------+
    bool IsSignalExpired(datetime expires_at)
    {
        return TimeGMT() >= expires_at;
    }

    //+------------------------------------------------------------------+
    //| Check if market is open for the symbol                            |
    //+------------------------------------------------------------------+
    bool IsMarketOpen(string symbol)
    {
        ENUM_SYMBOL_TRADE_MODE mode =
            (ENUM_SYMBOL_TRADE_MODE)SymbolInfoInteger(symbol, SYMBOL_TRADE_MODE);
        return mode != SYMBOL_TRADE_MODE_DISABLED;
    }

    //+------------------------------------------------------------------+
    //| Calculate P&L in pips for a closed trade                          |
    //+------------------------------------------------------------------+
    double CalculatePnlPips(string symbol, string direction,
                            double open_price, double close_price)
    {
        double point = SymbolInfoDouble(symbol, SYMBOL_POINT);
        int digits = (int)SymbolInfoInteger(symbol, SYMBOL_DIGITS);

        if(point <= 0) return 0;

        double pip_size = point;
        if(digits == 5 || digits == 3)
            pip_size = point * 10;

        double diff = close_price - open_price;
        if(direction == "sell")
            diff = -diff;

        return diff / pip_size;
    }

    //+------------------------------------------------------------------+
    //| Calculate P&L in R-multiples                                      |
    //+------------------------------------------------------------------+
    double CalculatePnlR(double open_price, double close_price, double stop_price)
    {
        double risk = MathAbs(open_price - stop_price);
        if(risk == 0) return 0;

        double pnl = close_price - open_price;
        return pnl / risk;
    }

    //--- Getters
    double DailyStartBalance()  const { return m_daily_start_balance; }
    double DailyRealizedPnl()   const { return m_daily_realized_pnl; }
    int    OpenPositions()      const { return m_open_atlas_positions; }
};

#endif // ATLAS_RISK_MQH
